package com.bcxin.tenant.domain.services.commands.contracts;

import com.bcxin.Infrastructures.commands.CommandAbstract;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.Infrastructures.utils.DateUtil;
import com.bcxin.tenant.domain.snapshots.ContractImportedItemSnapshot;
import lombok.Getter;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.Date;

@Getter
public class BatchImportContractCommand extends CommandAbstract {
    private final String organizationId;
    private final String path;
    private final Collection<ContractCommandItem> items;

    public BatchImportContractCommand(String organizationId, String path,
                                      Collection<ContractCommandItem> items) {
        this.organizationId = organizationId;
        this.path = path;
        this.items = items;
    }

    public static BatchImportContractCommand create(String organizationId, String path,
                                                    Collection<ContractCommandItem> items) {
        return new BatchImportContractCommand(organizationId, path, items);
    }

    @Override
    public void validate() {
        super.validate();

        if(!StringUtils.hasLength(this.getOrganizationId())) {
            throw new ArgumentTenantException("当前组织无效");
        }

        if(!StringUtils.hasLength(this.getPath())) {
            throw new ArgumentTenantException("上传的附件不能为空");
        }

        if (CollectionUtils.isEmpty(this.getItems())) {
            throw new ArgumentTenantException("待导入的合同列表不能为空");
        }

        this.getItems().forEach(ix -> {
            try {
                ix.validate();
            } catch (TenantExceptionAbstract ex) {
                ix.addError(ex.getMessage());
            }
        });
    }

    @Getter
    public static class ContractCommandItem extends CommandAbstract {
        private final ContractImportedItemSnapshot data;


        public ContractCommandItem(int index, String name, String aName, String bName,
                                   String idNum, String beginDate, String endDate) {
            this.data = ContractImportedItemSnapshot.create( index,name,aName,bName,idNum,beginDate,endDate);
        }

        public static ContractCommandItem create(int index,String name, String aName, String bName,
                                                 String idNum, String beginDate, String endDate) {
            return new ContractCommandItem(index, name, aName, bName, idNum, beginDate, endDate);
        }

        public Date getBeginDate() {
            return DateUtil.fromString(this.getData().getBeginDateText());
        }

        public Date getEndDate() {
            return DateUtil.fromString(this.getData().getEndDateText());
        }

        @Override
        public void validate() {
            super.validate();
            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getData().getName())) {
                    throw new ArgumentTenantException("合同名称不能为空");
                }
            });

            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getData().getAName())) {
                    throw new ArgumentTenantException("甲方名称不能为空");
                }
            });

            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getData().getBName())) {
                    throw new ArgumentTenantException("乙方名称不能为空");
                }
            });

            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getData().getBeginDateText())) {
                    throw new ArgumentTenantException("合同开始时间不能为空");
                }

                if (DateUtil.fromString(this.getData().getBeginDateText()) == null) {
                    throw new ArgumentTenantException("合同开始时间无效");
                }
            });

            checkParameter(() -> {
                if (!StringUtils.hasLength(this.getData().getEndDateText())) {
                    throw new ArgumentTenantException("合同结束时间不能为空");
                }

                if (DateUtil.fromString(this.getData().getEndDateText()) == null) {
                    throw new ArgumentTenantException("合同结束时间无效");
                }
            });

            checkParameter(() -> {
                Date beginDate = DateUtil.fromString(this.getData().getBeginDateText());
                Date endDate = DateUtil.fromString(this.getData().getEndDateText());
                if (beginDate != null && endDate != null) {
                    if (beginDate.after(endDate)) {
                        throw new ArgumentTenantException("合同开始时间不能大于结束时间");
                    }
                }
            });
        }
    }
}
