package com.bcxin.tenant.open.jdks.requests;

import com.alibaba.fastjson.annotation.JSONField;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventCategory;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventLevel;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventStatus;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import lombok.Data;
import lombok.Getter;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 通过监听回调来将数据加入到redis的set中实现; 总数量的计算
 * 同时解耦创建警情及警情处理的功能
 */
@Getter
public class CalculatePoliceIncidentsRequest extends RequestAbstract {

    private final Collection<PoliceIncidentSnapshot> policeIncidents;

    public Map<String,Collection<PoliceIncidentSnapshot>> getPendingSuperviseDepartPoliceIncidentsMap() {
        if (this.getPoliceIncidents() == null || this.getPoliceIncidents().isEmpty()) {
            return Collections.EMPTY_MAP;
        }

        return executeKeyBySuperviseDepartId(this.getPoliceIncidents(),
                Stream.of(PoliceEventStatus.None).collect(Collectors.toList())
        );
    }

    public Map<String,Collection<PoliceIncidentSnapshot>> getUnPendingSuperviseDepartPoliceIncidentsMap() {
        if (this.getPoliceIncidents() == null || this.getPoliceIncidents().isEmpty()) {
            return Collections.EMPTY_MAP;
        }

        return executeKeyBySuperviseDepartId(this.getPoliceIncidents(),
                Arrays.stream(PoliceEventStatus.values()).filter(ix -> ix != PoliceEventStatus.None)
                        .collect(Collectors.toList()
                        )
        );
    }


    private Map<String, Collection<PoliceIncidentSnapshot>> executeKeyBySuperviseDepartId(Collection<PoliceIncidentSnapshot> input,
                                                                                          Collection<PoliceEventStatus> statuses) {
        Map<String, Collection<PoliceIncidentSnapshot>> group = new HashMap<>();
        /**
         * 排除Sos的警情上报的数量
         */
        Collection<PoliceIncidentSnapshot> pendings =
                input.stream().filter(ix -> statuses.contains(ix.getPoliceEventStatus()))
                        .collect(Collectors.toList());
        for (PoliceIncidentSnapshot snapshot : pendings) {
            if (StringUtil.isEmpty(snapshot.getSuperviseDepartId())) {
                continue;
            }

            Collection<PoliceIncidentSnapshot> snapshots = group.get(snapshot.getSuperviseDepartId());
            if (snapshots == null) {
                snapshots = new ArrayList<>();

                group.put(snapshot.getSuperviseDepartId(), snapshots);
            }
            snapshots.add(snapshot);
        }

        return group;
    }

    public CalculatePoliceIncidentsRequest(Collection<PoliceIncidentSnapshot> policeIncidents) {
        this.policeIncidents = policeIncidents;
    }

    public static CalculatePoliceIncidentsRequest create(Collection<PoliceIncidentSnapshot> policeIncidents) {
        return new CalculatePoliceIncidentsRequest(policeIncidents);
    }

    /**
     * 用于处理警情上报的事件变更信息
     */
    @Data
    public static class PoliceIncidentSnapshot extends RequestAbstract {
        private Long id;

        @JSONField(name = "tenant_user_id")
        private String tenantUserId;

        @JSONField(name = "tenant_employee_id")
        private String tenantEmployeeId;

        @JSONField(name = "organization_id")
        private String organizationId;

        @JSONField(name = "security_station_id")
        private String securityStationId;

        @JSONField(name = "supervise_depart_id")
        private String superviseDepartId;

        @JSONField(name = "security_station_name")
        private String securityStationName;

        @JSONField(name = "code")
        private String code;

        @JSONField(name = "police_name")
        private String name;

        @JSONField(name = "contact")
        private String contact;

        @JSONField(name = "police_status")
        private PoliceEventStatus policeEventStatus;

        @JSONField(name = "police_level")
        private PoliceEventLevel policeEventLevel;

        @JSONField(name = "category")
        private PoliceEventCategory category;

        @JSONField(name = "policeEventType")
        private int policeEventType;

        @JSONField(name = "address")
        private String address;

        @JSONField(name = "reason")
        private String reason;

        @JSONField(name = "action_taken")
        private String actionTaken;

        @JSONField(name = "action_tendency")
        private String actionTendency;

        @JSONField(name = "processed_suggestion")
        private String processedSuggestion;

        @JSONField(name = "processed_locale_guidance")
        private String processedLocaleGuidance;

        @JSONField(name = "processed_user_id")
        private String processedUserId;

        @JSONField(name = "processed_user_name")
        private String processedUserName;

        @JSONField(name = "scene_url")
        private String sceneUrl;

        @JSONField(name = "processed_time")
        private Timestamp processedTime;

        @JSONField(name = "created_time")
        private Timestamp created_time;

        @JSONField(name = "emp_name")
        private String empName;

        @JSONField(name = "lon_lat")
        private LonlatLocationValue lonLat;

        public boolean isDeleted;

        @Data
        public static class LonlatLocationValue implements Serializable {
            private Double x;//经度
            private Double y;//维度

            public Double getLongitude() {
                return this.getY();
            }

            public Double getLatitude() {
                return this.getX();
            }
        }
    }
}
