package com.bcxin.tenant.open.backend.tasks.configs;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.snapshoots.RoomDispatchSnapshoot;
import com.bcxin.tenant.open.jdks.RollCallWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.BatchRollCallEmployeeRequest;
import com.bcxin.tenant.open.jdks.requests.RollCallBatchDispatchRequest;
import com.bcxin.tenant.open.jdks.requests.RollCallBatchEmployeeRoomActionRequest;
import com.bcxin.tenant.open.jdks.requests.RoomReplyRequest;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class KafkaComponent_RollCall {
    private static final Logger logger = LoggerFactory.getLogger(KafkaComponent_RollCall.class);
    private final RollCallWriterRpcProvider rollCallWriterRpcProvider;
    private final JsonProvider jsonProvider;

    public KafkaComponent_RollCall(RollCallWriterRpcProvider rollCallWriterRpcProvider, JsonProvider jsonProvider) {
        this.rollCallWriterRpcProvider = rollCallWriterRpcProvider;
        this.jsonProvider = jsonProvider;
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-roll-call-record",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ROLL_CALL_RECORD
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-roll-call-record")
    public void ackSyncRollCallRecord(List<ConsumerRecord<String, String>> records,
                                        Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效点名轮换数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        Collection<String> data =
                records.stream().map(ii -> ii.value()).collect(Collectors.toList());

        try {
            this.rollCallWriterRpcProvider.dispatch(BatchRollCallEmployeeRequest.create(data));
        } catch (Exception ex) {
            logger.error("点名轮换数据发生异常:{}",
                    data.stream().collect(Collectors.joining(",")),
                    ex);
            /**
             * 针对异常的情况直接跳过不要影响后面的功能
             */
            //allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-room-employee-action",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ROOM_EMPLOYEE_ACTION
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-room-employee-action")
    public void ackRoomAction(List<ConsumerRecord<String, String>> records,
                                      Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效点名轮换数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<RollCallBatchEmployeeRoomActionRequest.DispatchRollCallReplyRequest> data =
                    records.stream().map(ii -> {
                                RoomReplyRequest rpr =
                                this.jsonProvider.toObject(RoomReplyRequest.class, ii.value());
                                        return RollCallBatchEmployeeRoomActionRequest.DispatchRollCallReplyRequest.create(
                                                rpr.getId(),
                                                rpr.getActionType(),rpr.getEmployeeIds()
                                        );
                            })
                            .collect(Collectors.toList());

            this.rollCallWriterRpcProvider.dispatch(RollCallBatchEmployeeRoomActionRequest.create(data));
        } catch (Exception ex) {
            logger.error("点名轮换数据发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    /**
     * 针对发起调度的时候; referenceType为RollCall的时候;
     * 系统将数据加入到 TOPIC_DISPATCH_ROLL_CALL_EMPLOYEE_ROOM
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-roll-call-employee-room",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ROLL_CALL_EMPLOYEE_ROOM
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-roll-call-employee-room")
    public void ackRollCallRoomEmployee(List<ConsumerRecord<String, String>> records,
                              Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("督导点名记录信息:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<RollCallBatchDispatchRequest.RollCallDispatchRequest> data =
                    records.stream().map(ii -> {
                                RoomDispatchSnapshoot rpr =
                                        this.jsonProvider.toObject(RoomDispatchSnapshoot.class, ii.value());
                                if (rpr == null || CollectionUtils.isEmpty(rpr.getEmployeeIds())) {
                                    return null;
                                }
                                return RollCallBatchDispatchRequest.RollCallDispatchRequest.create(
                                        Long.parseLong(rpr.getReferenceNumber()),
                                        String.valueOf(rpr.getRoomId()),
                                        rpr.getEmployeeIds());
                            }).filter(ii -> ii != null)
                            .collect(Collectors.toList());

            this.rollCallWriterRpcProvider.dispatch(RollCallBatchDispatchRequest.create(data));
        } catch (Exception ex) {
            logger.error("督导点名处理发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    
}
