package com.bcxin.tenant.open.backend.tasks.configs;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.HotCacheRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import lombok.Data;
import lombok.Getter;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
public class KafkaComponent_RefreshRelative extends KafkaComponentAbstract {
    private static final Logger logger = LoggerFactory.getLogger(KafkaComponent.class);
    private final RdSyncRpcWriterProvider syncRpcWriterProvider;
    private final JsonProvider jsonProvider;
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;
    private final EmployeeWriterRpcProvider employeeWriterRpcProvider;
    private final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;

    private final HotCacheRpcProvider hotCacheRpcProvider;

    public KafkaComponent_RefreshRelative(RdSyncRpcWriterProvider syncRpcWriterProvider,
                                          JsonProvider jsonProvider,
                                          EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                          EmployeeWriterRpcProvider employeeWriterRpcProvider,
                                          SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                          DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider, HotCacheRpcProvider hotCacheRpcProvider) {
        super(jsonProvider,securityStationReaderRpcProvider,dispatchDataScopeRpcProvider,syncRpcWriterProvider);
        this.syncRpcWriterProvider = syncRpcWriterProvider;
        this.jsonProvider = jsonProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.employeeWriterRpcProvider = employeeWriterRpcProvider;
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.hotCacheRpcProvider = hotCacheRpcProvider;
    }

    /**
     * 人员/职员/驻勤/企业
     *
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-relative-changes-id",
            topics = {
                    KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES,
                    KafkaConstants.TOPIC_OBPM2_TENANT_ORGANIZATIONS,
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION,
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION_PERSON
            }, groupId = "${spring.kafka.consumer.group-id}-relative-changes")
    public void ackRelativeChangesInformationListener(List<ConsumerRecord<String, String>> records,
                                                      Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.error("无效订阅数据:{}", records);
            return;
        }

        DispatchDataType dataType = DispatchDataType.Employee;
        Collection<String> topics = records.stream()
                .map(ix -> ix.topic()).distinct()
                .collect(Collectors.toList());

        boolean allowed2CommitAtFinial = true;
        try {
            for (String topic : topics) {
                try {
                    Collection<ConsumerRecord<String, String>> selectedRecords =
                            records.stream()
                                    .filter(ii -> ii.topic().equalsIgnoreCase(topic))
                                    .collect(Collectors.toList());
                    if (CollectionUtils.isEmpty(selectedRecords)) {
                        continue;
                    }

                    Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
                    Collection<DebeziumSnapshot> snapshots
                            = selectedRecords.stream()
                            .map(ix -> this.jsonProvider.toObject(DebeziumSnapshot.class, ix.value()))
                            .collect(Collectors.toList());
                    switch (topic) {
                        case KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES:
                            dataType = DispatchDataType.Employee;
                            syncKeyItems.addAll(doExecuteEmployeeChange(snapshots));
                            syncKeyItems.addAll(doExecuteNewEmployeeForCompany(snapshots));
                            break;
                        case KafkaConstants.TOPIC_OBPM2_TENANT_ORGANIZATIONS:
                            dataType = DispatchDataType.Company;
                            syncKeyItems.addAll(doExecuteOrganization(snapshots));
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION:
                            dataType = DispatchDataType.Station;
                            syncKeyItems.addAll(doExecuteStationChange(snapshots));
                            syncKeyItems.addAll(doExecuteNewStationForCompany(snapshots));
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION_PERSON:
                            dataType = DispatchDataType.StationPerson;
                            syncKeyItems.addAll(doExecuteStationPerson(snapshots));
                            break;
                        default:
                            throw new NotSupportTenantException(String.format("KafkaComponent_RefreshRelative.ackRelativeChangesInformationListener 不支持该主题(%s)信息", topic));
                    }

                    /*
                    if(CollectionUtils.isEmpty(syncKeyItems)) {
                        logger.error("KafkaComponent_RefreshRelative map items size is empty;topic={};content={};",
                                topic, selectedRecords.stream().map(ii -> ii.key()).collect(Collectors.joining(",")));
                    }
                     */

                    syncKeyItems.forEach(item -> {
                        syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(item.getDataType(),item.getIds()));
                        switch (item.getDataType()) {
                            case Employee -> {
                                this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Employee,item.getIds()));
                            }
                            case Company -> {
                                this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Organization,item.getIds()));
                            }
                            case Station -> {
                                this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Station,item.getIds()));
                            }
                        }
                    });
                } catch (Exception ex) {
                    allowed2CommitAtFinial = false;
                    ex.printStackTrace();
                    logger.error("KafkaComponent_RefreshRelative: failed to consumer data with topic={}", topic, ex);
                }
            }
        } catch (Exception ex) {
            allowed2CommitAtFinial = false;
            ex.printStackTrace();
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    /**
     * 针对职员信息入离职信息发生变更(离职/复职) 需要同步修改驻勤/公司中的保安员数量
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteEmployeeChange(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> cIds = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return false;
            }

            if (ix.getAfter() == null) {
                return true;
            }

            return ix.getAfter().getInteger("status") != ix.getBefore().getInteger("status");
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("organization_id");
            }

            return ix.getAfter().getString("organization_id");
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(cIds)) {
            logger.warn("找不到变更的数据-主题为:{}-数据-{}",
                    KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES,
                    snapshots.stream().map(ix -> {
                        if (ix.getBefore() != null) {
                            return ix.getBefore().getString("organization_id");
                        }

                        return ix.getAfter().getString("organization_id");
                    }).collect(Collectors.joining(";"))
            );
            return syncKeyItems;
        }

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Company, cIds));

        Collection<String> eIds2 = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return false;
            }

            if (ix.getAfter() == null) {
                return true;
            }

            return ix.getAfter().getInteger("status") != ix.getBefore().getInteger("status");
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("id");
            }

            return ix.getAfter().getString("id");
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(cIds)) {
            logger.warn("找不到变更的数据-主题为:{}-数据-{}",
                    KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES,
                    snapshots.stream().map(ix -> {
                        if (ix.getBefore() != null) {
                            return ix.getBefore().getString("id");
                        }

                        return ix.getAfter().getString("id");
                    }).collect(Collectors.joining(";"))
            );

            return syncKeyItems;
        }
        Collection<String> esIds =
                this.employeeWriterRpcProvider.getStationIdsByEmployeeIds(eIds2);
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Station, esIds));

        return syncKeyItems;
    }

    /**
     * 针对新增人员的时候, 刷新公司的人员数量
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteNewEmployeeForCompany(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> cIds = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return true;
            }

            return false;
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("organization_id");
            }

            return ix.getAfter().getString("organization_id");
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(cIds)) {
            logger.warn("找不到新增的职员数据-主题为:{}", KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES);
            return syncKeyItems;
        }

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Company, cIds));

        return syncKeyItems;
    }

    /**
     * 针对修改驻勤信息的时候, 变更人员的最新驻勤及公司的驻勤数量
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteStationChange(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> s2Ids = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return false;
            }

            if (ix.getAfter() == null) {
                return true;
            }

            try {
                return (!StringUtil.isEqual(
                        ix.getAfter().getString("ITEM_attendanceSiteState"),
                        ix.getBefore().getString("ITEM_attendanceSiteState"))
                        ||
                        !StringUtil.isEqual(
                                ix.getAfter().getString("ITEM_attendanceSiteName"),
                                ix.getBefore().getString("ITEM_attendanceSiteName"))
                        ||
                        !StringUtil.isEqual(
                                ix.getAfter().getString("ITEM_officePoliceAddressID"),
                                ix.getBefore().getString("ITEM_officePoliceAddressID"))
                        ||
                        !StringUtil.isEqual(
                                ix.getAfter().getString("ITEM_attendanceSiteState"),
                                ix.getBefore().getString("ITEM_attendanceSiteState"))
                        ||
                        !StringUtil.isEqual(
                                String.valueOf(ix.getAfter().getString("ITEM_attendanceStartDate")),
                                String.valueOf(ix.getBefore().getString("ITEM_attendanceStartDate")))
                        ||
                        !StringUtil.isEqual(
                                String.valueOf(ix.getAfter().getString("ITEM_attendanceEndDate")),
                                String.valueOf(ix.getBefore().getString("ITEM_attendanceEndDate"))
                        )
                        ||
                        !StringUtil.isEqual(
                                ix.getAfter().getString("ITEM_principal"),
                                ix.getBefore().getString("ITEM_principal"))
                );
            }
            catch (Exception ex) {
                logger.error("转换发生异常", ex);
            }

            return true;
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("ID");
            }

            return ix.getAfter().getString("ID");
        }).collect(Collectors.toList());
        syncKeyItems.addAll(extractStationPrincipals(snapshots));

        if (CollectionUtils.isEmpty(s2Ids)) {
            logger.warn("找不到变更的数据-主题为:{}", KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION);
            return syncKeyItems;
        }

        Collection<String> organizationIds =
                this.securityStationReaderRpcProvider.getOrganizationIdsByStationIds(s2Ids);
        Collection<String> employeeIds2 =
                this.employeeReaderRpcProvider.getEmployeeIdsByStationIds(s2Ids);

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Company, organizationIds));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Employee, employeeIds2));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Member, employeeIds2));

        return syncKeyItems;
    }

    private  Collection<SyncKeyItem> extractStationPrincipals(Collection<DebeziumSnapshot> snapshots) {
        Set<SyncKeyItem> syncKeyItems = new HashSet<>();
        /**
         * 驻勤点负责人
         */
        Collection<String> s3Principals = snapshots.stream().flatMap(ix -> {
            Collection<String> selectedPrincipals = new ArrayList<>();
            if (ix.getBefore() != null) {
                String pId = ix.getBefore().getString("ITEM_principal");
                if (StringUtils.hasLength(pId)) {
                    selectedPrincipals.add(pId);
                }
            }

            if (ix.getAfter() != null) {
                String pId = ix.getAfter().getString("ITEM_principal");
                if (StringUtils.hasLength(pId)) {
                    selectedPrincipals.add(pId);
                }
            }
            return selectedPrincipals.stream();
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(s3Principals)) {
            return Collections.EMPTY_LIST;
        }

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Employee, s3Principals.stream().distinct().collect(Collectors.toList())));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Member, s3Principals.stream().distinct().collect(Collectors.toList())));

        return syncKeyItems;
    }

    /**
     * 针对新增驻勤信息的时候, 变更其他的驻勤数量
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteNewStationForCompany(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> s2Ids = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return true;
            }

            if (ix.getAfter() == null) {
                return true;
            }

            try {
                String attendanceSiteStateColumnName = ix.getAfter().keySet().stream().filter(ii ->
                        ii.equalsIgnoreCase("ITEM_attendanceSiteState")
                ).findFirst().orElse(null);

                String contractStartDateColumnName = ix.getAfter().keySet().stream().filter(ii ->
                        ii.equalsIgnoreCase("ITEM_contractStartDate")
                ).findFirst().orElse(null);
                String contractEndDateColumnName = ix.getAfter().keySet().stream().filter(ii ->
                        ii.equalsIgnoreCase("ITEM_contractEndDate")
                ).findFirst().orElse(null);
                Collection<String> keyColumns =
                        Stream.of(attendanceSiteStateColumnName,
                                        contractStartDateColumnName,
                                        contractEndDateColumnName)
                                .filter(ii -> !StringUtil.isEmpty(ii))
                                .collect(Collectors.toList());
                if (CollectionUtils.isEmpty(keyColumns)) {
                    return true;
                }

                for (String col : keyColumns) {
                    if (!StringUtil.isEqual(ix.getAfter().getString(col), ix.getBefore().getString(col))) {
                        return true;
                    }
                }
            } catch (Exception ex) {
                logger.error("parse error while compare attendance site column value; for this case; we make company changed.", ex);
                return true;
            }

            return false;
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("ID");
            }

            return ix.getAfter().getString("ID");
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(s2Ids)) {
            logger.warn("找不到新增的驻勤点数据-主题为:{}", KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION);
            return syncKeyItems;
        }

        Collection<String> organizationIds =
                this.securityStationReaderRpcProvider.getOrganizationIdsByStationIds(s2Ids);

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Company, organizationIds));

        syncKeyItems.addAll(extractStationPrincipals(snapshots));

        return syncKeyItems;
    }

    /**
     * 新增驻勤人员的时候, 需要刷新驻勤中的人员总数
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteStationPerson(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> eIds = snapshots.stream().map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("ITEM_securityId");
            }

            return ix.getAfter().getString("ITEM_securityId");
        }).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(eIds)) {
            logger.warn("找不到变更的数据-职员-主题为:{}", KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION_PERSON);
            return syncKeyItems;
        }

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Employee, eIds));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Member, eIds));

        Collection<String> sIds = snapshots.stream().map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("ITEM_attendanceSiteId");
            }

            return ix.getAfter().getString("ITEM_attendanceSiteId");
        }).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(sIds)) {
            logger.warn("找不到变更的数据-驻勤点-主题为:{}", KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION_PERSON);
            return syncKeyItems;
        }

        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Station, sIds));

        return syncKeyItems;
    }


    /**
     * 变更其他名称的时候, 需要刷新到人员和驻勤的企业名称
     *
     * @param snapshots
     * @return
     */
    private Collection<SyncKeyItem> doExecuteOrganization(Collection<DebeziumSnapshot> snapshots) {
        Collection<SyncKeyItem> syncKeyItems = new ArrayList<>();
        Collection<String> oIds = snapshots.stream().filter(ix -> {
            if (ix.getBefore() == null) {
                return false;
            }

            if (ix.getAfter() == null) {
                return true;
            }

            return true;
            //return !StringUtil.isEqual(ix.getAfter().getString("name"), ix.getBefore().getString("name"));
        }).map(ix -> {
            if (ix.getBefore() != null) {
                return ix.getBefore().getString("id");
            }

            return ix.getAfter().getString("id");
        }).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(oIds)) {
            logger.warn("找不到变更的数据-主题为:{}", KafkaConstants.TOPIC_OBPM2_TENANT_ORGANIZATIONS);
            return syncKeyItems;
        }

        Collection<String> employeeIds =
                this.employeeReaderRpcProvider.getEmployeeIdsByOrganizationIds(oIds);
        Collection<String> stationIds =
                this.securityStationReaderRpcProvider.getSecurityStationIdsByOrganizationIds(oIds);
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Station, stationIds));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Employee, employeeIds));
        syncKeyItems.add(SyncKeyItem.create(DispatchDataType.Member, employeeIds));

        return syncKeyItems;
    }

    @Data
    public static class DebeziumSnapshot {
        private JSONObject before;
        private JSONObject after;
        private long ts_ms;
    }

    @Getter
    public static class SyncKeyItem {
        private final DispatchDataType dataType;
        private final Collection<String> ids;

        public SyncKeyItem(DispatchDataType dataType, Collection<String> ids) {
            this.dataType = dataType;
            this.ids = ids;
        }


        public static SyncKeyItem create(DispatchDataType dataType, Collection<String> ids) {
            return new SyncKeyItem(dataType, ids);
        }
    }
}
