package com.bcxin.tenant.open.backend.tasks.components;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.jdks.DeviceWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.BatchDeviceLocationRequest;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

@Component
public class OnMqttMessageCallback implements MqttCallback {
    private static final Logger logger = LoggerFactory.getLogger(DeviceMqttCommandLineRunner.class);
    private final JsonProvider jsonProvider;
    private final DeviceWriterRpcProvider deviceWriterRpcProvider;
    private final static Collection<BatchDeviceLocationRequest.DeviceLocationRequest> _globalBatchLocationRequests = (new ArrayList<>());

    public OnMqttMessageCallback(JsonProvider jsonProvider, DeviceWriterRpcProvider deviceWriterRpcProvider) {
        this.jsonProvider = jsonProvider;
        this.deviceWriterRpcProvider = deviceWriterRpcProvider;

        /**
         * 每20秒检查一次经纬度信息
         */
        Executors.newScheduledThreadPool(1).scheduleAtFixedRate(() -> {
            this.flush();
        }, 10, 30, TimeUnit.SECONDS);
    }

    @Override
    public void connectionLost(Throwable cause) {
        // 连接丢失后，一般在这里面进行重连
        logger.error("连接断开，可以做重连");
    }

    @Override
    public void messageArrived(String topic, MqttMessage message) throws Exception {
        String locationJson = null;
        try {
            // subscribe后得到的消息会执行到这里面
            locationJson = new String(message.getPayload());
            JSONObject location =
                    this.jsonProvider.toObject(JSONObject.class, locationJson);
            double lat_x1e6 = location.getLongValue("lat_x1e6") / 10_00000.0;
            double lon_x1e6 = location.getLongValue("lon_x1e6") / 10_00000.0;

            CoordinateConverter.AMap aMap = CoordinateConverter.transform(lon_x1e6, lat_x1e6);
            // double[] convertedLonlat = CoordinateConverter.wgs84ToGcj02(lat_x1e6,lon_x1e6);
            lat_x1e6 = aMap.getLatitude();
            lon_x1e6 = aMap.getLongitude();

            String dispatchNo = location.getString("senderAcc");
            long time = Instant.now().getEpochSecond();
            try {
                time = location.getLongValue("time");
            } catch (Exception ex) {
                logger.error("获取time值发生异常:{}", locationJson);
            }

            String senderUID = location.getString("senderUID");

            synchronized (OnMqttMessageCallback.class) {
                _globalBatchLocationRequests.add(
                        BatchDeviceLocationRequest.DeviceLocationRequest.create(
                                senderUID,
                                time,
                                dispatchNo,
                                lon_x1e6,
                                lat_x1e6
                        ));

                if (_globalBatchLocationRequests.size() > 100) {
                    flush();
                } else {
                    logger.error("记录经纬度上传信息:dispatchNo={};lon={};lat={};size={};time={};原始的坐标信息:{}", dispatchNo, lon_x1e6, lat_x1e6,
                            _globalBatchLocationRequests.size(), time, locationJson);
                }
            }
        } finally {
            logger.error("messageArrived 收到mqtt队列主题={};信息:{}", topic, locationJson);
        }
    }

    @Override
    public void deliveryComplete(IMqttDeliveryToken token) {
        logger.error("deliveryComplete---------" + token.isComplete());
    }

    public  void flush() {
        synchronized (OnMqttMessageCallback.class) {
            if (_globalBatchLocationRequests.size() > 0) {
                Collection<BatchDeviceLocationRequest.DeviceLocationRequest> items = _globalBatchLocationRequests;
                BatchDeviceLocationRequest batchDeviceLocationRequest = BatchDeviceLocationRequest.create(items);

                String batchJson = this.jsonProvider.getJson(batchDeviceLocationRequest);

                this.deviceWriterRpcProvider.batch(batchDeviceLocationRequest);
                _globalBatchLocationRequests.removeAll(items);

                logger.error("批量提交设备经纬度信息到服务器:" + batchJson);
            }
        }
    }
}