package com.bcxin.web.commons.components;

import com.bcxin.saas.core.components.DistributedCacheProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.utils.RetryUtil;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.*;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

@Slf4j
@Component
public class RedisDistributedCacheProviderImpl implements DistributedCacheProvider {

    private final RedisTemplate redisTemplate;
    private final RedissonClient redissonClient;

    public RedisDistributedCacheProviderImpl(RedisTemplate redisTemplate, RedissonClient redissonClient) {
        this.redisTemplate = redisTemplate;
        this.redissonClient = redissonClient;
    }

    @Override
    public <T> T get(String key) {
        try {
            Object data = this.redisTemplate.opsForValue().get(key);
            if (data == null) {
                return null;
            }

            return (T) data;
        } catch (Exception ex) {
            log.error(String.format("从Redis获取(%s)信息失败", key), ex);
        }

        return null;
    }

    @Override
    public <T> T get(String key, Supplier<T> callback, long expires) {
        T data = this.get(key);

        if (data != null) {
            return data;
        }

        if (callback == null) {
            throw new SaasBadException("无效回调函数");
        }
        data = callback.get();
        this.put(key, data, expires);

        return data;
    }

    @Override
    public <T> void put(String key, T data) {
        try {
            this.redisTemplate.opsForValue().set(key, data);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }

    @Override
    public <T> void put(String key, T data, long expires) {
        try {
            this.redisTemplate.opsForValue().set(key, data,expires, TimeUnit.SECONDS);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }

    @Override
    public <T> void putForSet(String key, Collection<T> data) {
        try {
            this.redisTemplate.delete(key);
            SetOperations setOperations = this.redisTemplate.opsForSet();
            Object[] arrayD = data.toArray(new Object[data.size()]);
            setOperations.add(key, arrayD);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }
    @Override
    public <T> void putForSetWithExpired(String key, Collection<T> data, long expires) {
        try {
            this.redisTemplate.delete(key);
            SetOperations setOperations = this.redisTemplate.opsForSet();
            Object[] arrayD = data.toArray(new Object[data.size()]);
            setOperations.add(key, arrayD);

           this.redisTemplate.expire(key, expires, TimeUnit.SECONDS);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }

    @Override
    public <T> void addToSet(String key, T data) {
        try {
            SetOperations setOperations = this.redisTemplate.opsForSet();
            setOperations.add(key, data);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }

    @Override
    public long getSetSize(String key) {
        return this.redisTemplate.opsForSet().size(key);
    }

    @Override
    public <T> Collection<T> getRandomFromSet(String key, int count) {
        Collection<T> dt = this.redisTemplate.opsForSet().randomMembers(key, count);
        return dt;
    }

    @Override
    public <T> Collection<T> getFromSet(String key) {
        return this.redisTemplate.opsForSet().members(key);
    }

    @Override
    public <T> void removeFromSet(String key, T data) {
        try {
            SetOperations setOperations = this.redisTemplate.opsForSet();
            setOperations.remove(key, data);
        } catch (Exception ex) {
            log.error(String.format("Redis存储(%s)信息失败", key), ex);
        }
    }

    @Override
    public void clear(String prefix) {
        RetryUtil.execute(() -> {
            Collection<String> keys = keys(prefix);
            this.redisTemplate.delete(keys);
            return null;
        });
    }

    @Override
    public void remove(String key) {
        this.redisTemplate.delete(key);
    }

    @Override
    public Collection<String> keys(String prefix) {
        Set<String> keys = this.redisTemplate.keys(String.format("%s*", prefix));

        return keys;
    }

    @Override
    public void remove(Collection<String> keys) {
        RetryUtil.execute(() -> {
            this.redisTemplate.delete(keys);
            return null;
        });
    }

    @Override
    public <T> void batchPut(Map<String, T> data) {
        if (CollectionUtils.isEmpty(data)) {
            return;
        }

        RedisSerializer keySerializer = this.redisTemplate.getKeySerializer();
        RedisSerializer valueSerializer = this.redisTemplate.getValueSerializer();
        RetryUtil.execute(() -> {
            this.redisTemplate.executePipelined(new RedisCallback<T>() {
                @Override
                public T doInRedis(RedisConnection redisConnection) throws DataAccessException {

                    data.keySet().parallelStream().forEach(key -> {
                        T value = data.get(key);
                        if(value!=null) {
                            redisConnection.set(keySerializer.serialize(key), valueSerializer.serialize(value));
                        }else {
                            redisConnection.del(keySerializer.serialize(key));
                        }
                    });

                    return null;
                }
            });

            return null;
        }, 10);

    }

    @Override
    public boolean executeByLock(String key,Runnable runnable) {
        RLock rLock = redissonClient.getLock(getRedissonLockKey(key));

        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        try {
            rLock.lock();
            runnable.run();
        } finally {
            rLock.unlock();
        }
        stopWatch.stop();

        return true;
    }

    @Override
    public boolean lock(String key, int expiredInSeconds) {
        RLock lock = this.redissonClient.getLock(getRedissonLockKey(key));
        lock.lock(expiredInSeconds, TimeUnit.SECONDS);

        return true;
    }

    @Override
    public void release(String key) {
        RLock lock = this.redissonClient.getLock(getRedissonLockKey(key));
        lock.unlock();
    }

    private static String getRedissonLockKey(String key) {
        return String.format("com:redisson:lock:%s", key.toLowerCase());
    }
}
