package com.teemlink.qm.base.web.support;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author ZhanBo-ZJB
 * 映射请求数据至Action属性支持类
 * 支持数组,对象,泛型,参数化类型(近支持Set,List)的注入
 * Created by ZhanBo-ZJB on 2017/10/22.
 */
public class MapParamsToActionSupport implements DataTransportationSupport<HttpServletRequest,Object> {

    private static final Logger log = LoggerFactory.getLogger(MapParamsToActionSupport.class);

    @Override
    public void handle(HttpServletRequest dataSource,Object target) {
        //获取所有参数名称
        for (Enumeration<String> paramNames = dataSource.getParameterNames(); paramNames.hasMoreElements();) {
            //获取参数名称
            String paramName = paramNames.nextElement();

            try{

                //获取参数值
                String value = dataSource.getParameter(paramName).toString();

                //获取参数数组值
                String[] arrayValue = dataSource.getParameterValues(paramName);

                if(paramName.contains(".")) {
                    setObjectField(target,paramName,value,arrayValue);
                } else {
                    setBaseField(target,paramName,value,arrayValue);
                }

            }catch (Exception e) {
                //log.debug("{}", "映射属性"+paramName+"错误!属性不存在!");
                continue;
            }

        }
    }

    /**
     *
     * 反射设置基本属性
     * @param obj
     * @param paramName
     * @param paramValue
     * @param arrayValue
     * @throws Exception
     */
    protected void setBaseField(Object obj,String paramName,String paramValue,String[] arrayValue) throws Exception {
        //获取参数所对应的对象属性
        Field field = getField(obj.getClass(),paramName);

        //设置属性可操作
        field.setAccessible(true);

        //获取参数类型
        Class type = field.getType();

        //处理字符串日期转换
        ConvertUtils.register(new Converter() {

            @Override
            public Object convert(Class type, Object value) {

                if(value == null) {
                    return null;
                }
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                try {
                    return simpleDateFormat.parse(value.toString());
                } catch (ParseException e) {
                    simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");

                    try {
                        return simpleDateFormat.parse(value.toString());
                    }catch (ParseException parseException) {
                        return null;
                    }

                }
            }
        }, Date.class);

        //获取泛型参数
        Type genericType = field.getGenericType();

        //判断是否为泛型类型
        if(genericType != null && genericType instanceof ParameterizedType) {
            //处理泛型,仅支持(Set,List)
            Type argumentType = ((ParameterizedType) genericType).getActualTypeArguments()[0];

            if(((Class)field.getType()).isAssignableFrom(List.class)){
                List collection = new ArrayList();
                for(String e:arrayValue) {
                    collection.add(ConvertUtils.convert(e,(Class)argumentType));
                }
                field.set(obj,collection);
            }else if(((Class)field.getType()).isAssignableFrom(Set.class)){
                Set collection = new HashSet();
                for(String e:arrayValue) {
                    collection.add(ConvertUtils.convert(e,(Class)argumentType));
                }
                field.set(obj,collection);
            }else if (field.getType().equals(Collection.class)) {
                Collection collection = new ArrayList();
                for(String e:arrayValue) {
                    collection.add(ConvertUtils.convert(e,(Class)argumentType));
                }
                field.set(obj,collection);
            }

        }else{
            //如果属性为数组
            if( type.isArray() ) {
                //设置数组属性
                field.set(obj, ConvertUtils.convert(arrayValue,type));
            }else{
                //设置属性
                field.set(obj, ConvertUtils.convert(paramValue,type));
            }
        }
    }

    /**
     * 反射设置对象属性
     * @param obj
     * @param paramName
     * @param paramValue
     * @param arrayValue
     * @throws Exception
     */
    protected void setObjectField(Object obj,String paramName,String paramValue,String[] arrayValue) throws Exception {
        //切割属性
        String[] strs = paramName.split("\\.");

        //获取对象名称
        String objName = strs[0];

        //获取对象属性名称
        String objFieldName = strs[1];

        //获取参数所对应的对象属性
        Field field = getField(obj.getClass(),objName);

        //设置属性可操作
        field.setAccessible(true);

        //获取参数类型
        Class type = field.getType();

        //相等则表示该参数是泛型属性,使用特定方法(getGenericType)获取其真正的参数化后的类型
        if(field.getGenericType() instanceof TypeVariable) {
            Method method = obj.getClass().getMethod("getGenericType",Integer.class);
            type = (Class)method.invoke(obj,new Integer(0));
        }

        //判断被设置对象该属性是否为空
        if(field.get(obj) != null) {
            //如果不为空
            Object o = field.get(obj);

            setBaseField(o,objFieldName,paramValue,arrayValue);
        }else{

            try{
                //如果为空,为其初始化属性
                field.set(obj,type.newInstance());

                Object o = field.get(obj);

                setBaseField(o,objFieldName,paramValue,arrayValue);
            }catch (Exception e) {
                log.debug("{}", paramName+"映射错误!属性不存在");
            }

        }
    }

    /**
     * 通过属性名称获取class对象的属性对象(通过回调的方式到父类中寻找属性)
     * @param
     * @return
     */
    private Field getField(Class clazz, String name ){

        Field field = null;

        try{
            field = clazz.getDeclaredField(name);

        }catch (Exception e){
            field = getField(clazz.getSuperclass(),name);

        }

        field.setAccessible(true);
        return field;

    }
}
