package com.teemlink.qm.base.web.interceptor;

import com.teemlink.qm.base.web.NoValidate;
import com.teemlink.qm.base.web.OnlyValidateParams;
import com.teemlink.qm.base.web.support.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @author ZhanBo-ZJB
 * Action拦截器主要完成以下6件事情
 * 1.映射请求参数到Action的对应属性
 * 2.验证Action中属性数据是否正确
 * 3.调用set方法覆盖Action中属性
 * 4.处理上传文件,映射到Action的相应的File类型属性中
 * 5.映射request请求参数和Action中属性映射至ModelAndView中作为返回结果
 * 6.调用get方法覆盖Model中某些属性
 * Created by ZhanBo-ZJB on 2017/10/18.
 */
public class ActionInterceptor implements HandlerInterceptor{

    protected static final Logger LOG = LoggerFactory.getLogger(ActionInterceptor.class);

    private final String REDIRECT = "redirect:";

    /**
     * 属性映射支持模仿struts
     */
    private DataTransportationSupport mapParamToActionSupport = new MapParamsToActionSupport();

    /**
     * 参数验证支持JSR303,提供@JSR303注解用于标记具体的需要验证的Model,提供@NoValidate标记不需要验证的对象,提供@OnlyValidateParams标注仅需要验证对象的哪些属性
     */
    private DataTransportationSupport validateParamsSupport = new ValidateSupport();

    /**
     * 映射属性到model中,提供MapInModel标记Action中哪些属性需要映射到Model中作为结果返回
     */
    private DataTransportationSupport mapFieldToModelAndViewSupport = new MapFieldToModelAndViewSupport();

    /**
     * 映射请求参数到Model中
     */
    private DataTransportationSupport mapParamsToModelAndViewSupport = new MapParamsToModelAndView();

    /**
     * 提供Get方法调用支持
     */
    private DataTransportationSupport getMethodInvokeSupport = new GetMethodInvokeSupport();

    /**
     * 提供Set方法调用支持
     */
    private DataTransportationSupport setMethodInvokeSupport = new SetMethodInvokeSupport();

    /**
     * 提供上传文件映射支持
     */
    private DataTransportationSupport mapFileToActionSupport = new MapFileToActionSupport();

    @Override
    public boolean preHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o) throws Exception {

        //判断请求对应的是否为Action中的方法
        if( o instanceof HandlerMethod ) {

            //获取Action对象
            Object controller = ((HandlerMethod)o).getBean();

            //参数映射
            mapParamToActionSupport.handle(httpServletRequest,controller);

            //调用set方法并覆盖
            setMethodInvokeSupport.handle(httpServletRequest,controller);

            //文件上传
            mapFileToActionSupport.handle(httpServletRequest,controller);

            //判断方法是否有OnlyValidateParams注解,有则获取需要验证的参数,没有则会验证对象里面所有标记验证注解的属性
            List<String> validateParams = null;
            OnlyValidateParams validateAnnotation = ((HandlerMethod)o).getMethod().getAnnotation(OnlyValidateParams.class);
            if(validateAnnotation != null) {
                validateParams = Arrays.asList(validateAnnotation.params());
            }

            //判断方法是否有NoValidate注解，有则获取不需要验证的对象名称
            List<String> noValidateObjects = null;
            NoValidate noValidate = ((HandlerMethod)o).getMethod().getAnnotation(NoValidate.class);
            if( noValidate != null) {
                noValidateObjects = Arrays.asList(noValidate.objectNames());
            }

            //获取所有请求参数
            Enumeration<String> enumeration = httpServletRequest.getParameterNames();
            Map<String,Set<String>> objs = new HashMap<String, Set<String>> ();
            //迭代参数
            while (enumeration.hasMoreElements()) {
                String paramName = enumeration.nextElement().toString();
                if(paramName.contains(".")){
                    //切割属性
                    String[] strs = paramName.split("\\.");

                    //获取对象名称
                    String objName = strs[0];

                    //忽略不需要验证的对象
                    if(noValidateObjects != null && noValidateObjects.size() >0 && noValidateObjects.contains(objName)) {
                        continue;
                    }

                    //为真,则仅对对象部分属性进行验证
                    if( validateParams != null && validateParams.size() > 0){
                        if(validateParams.contains(paramName)) {
                            Set<String> fields = objs.get(objName);
                            if(fields == null) {
                                fields = new HashSet<String>();
                                fields.add(strs[1]);
                                objs.put(objName,fields);
                            }else{
                                fields.add(strs[1]);
                            }
                        }
                    }else {
                        objs.put(objName,null);
                    }
                }
            }
            //验证数据
            validateParamsSupport.handle(objs,controller);

            //反射调用getParams方法初始化数据
            Method getParamsMethod = controller.getClass().getMethod("getParams",HttpServletRequest.class);
            try{
                getParamsMethod.invoke(controller,httpServletRequest);
            }catch (InvocationTargetException e) {
                LOG.debug("{}", e.getMessage());
            }

        }

        return true;
    }

    @Override
    public void postHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, ModelAndView modelAndView) throws Exception {

        if(modelAndView != null) {
            if( o instanceof HandlerMethod ) {

                Object controller = ((HandlerMethod)o).getBean();
                if(!modelAndView.getViewName().contains(REDIRECT)){
                    //映射参数到ModelAndView
                    mapParamsToModelAndViewSupport.handle(httpServletRequest,modelAndView);

                    //映射属性至ModelAndView
                    mapFieldToModelAndViewSupport.handle(controller,modelAndView);

                    //调用get方法并覆盖
                    getMethodInvokeSupport.handle(controller,modelAndView);
                }

            }
        }

    }

    @Override
    public void afterCompletion(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, Exception e) throws Exception {

    }
}
