import Node from "./Node";
import PaintElement from "./PaintElement";
import Resources from "../utility/Resources";
import StringUtil from "../utility/StringUtil";

class AutoNode extends Node {
  constructor(diagram) {
    super(diagram);

    /**
     * 自动审批时间编辑模式(默认为设计模式)
     */
    this.autoAuditTimeEditMode = AutoNode.AUTO_AUDIT_EDIT_MODE_DESIGN;

    /**
     * 指定自动审批日期脚本
     */
    this.auditDateTimeScript = "";

    /**
     * 是否聚合节点
     */
    this.isgather = false;

    /**
     * 是否分散节点
     */
    this.issplit = false;

    /**
     * 自动审批类型
     */
    this.autoAuditType = 0;

    /**
     * 指定自动审批日期
     */
    this.auditDateTime = "";

    /**
     * 滞后时间(到达以后多少时间之后自动审批)
     */
    this.delayTime = "";

    /**
     * 滞后天数
     */
    this.delayDay = "";

    /**
     * 滞后小时数
     */
    this.delayHour = "";

    /**
     * 滞后分钟数
     */
    this.delayMinute = "";

    this.splitStartNode = ""; // 分散节点
  }

  getClassName() {
    return "cn.myapps.runtime.workflow.element.AutoNode";
  }

  getAuditDateTime(runner) {
    const calendar = Calendar.getInstance();

    if (autoAuditTimeEditMode == AUTO_AUDIT_EDIT_MODE_CODE && !StringUtil.isBlank(auditDateTimeScript)) {
      const obj = runner.run("AutoNode [name=" + name + "] [id=" + id + "] auditDateTimeScript", StringUtil.dencodeHTML(auditDateTimeScript));
      if (obj instanceof Date) {
        calendar.setTime(obj);
      } else if (obj instanceof String) {
        const date = DateUtil.parseDate(obj, "yyyy-MM-dd HH:mm:ss");
        calendar.setTime(date);
      }
    } else {
      switch (autoAuditType) {
        case AUTO_AUDIT_TYPE_IMMEDIATELY:
          break;
        case AUTO_AUDIT_TYPE_SPECIFY:
          const date = DateUtil.parseDate(this.auditDateTime, "yyyy-MM-dd HH:mm:ss");
          calendar.setTime(date);
          break;
        case AUTO_AUDIT_TYPE_DELAY:
          if (!StringUtil.isBlank(delayDay)) {
            calendar.add(Calendar.DATE, Integer.parseInt(delayDay));
          }
          if (!StringUtil.isBlank(delayHour)) {
            calendar.add(Calendar.HOUR, Integer.parseInt(delayHour));
          }
          if (!StringUtil.isBlank(delayMinute)) {
            calendar.add(Calendar.MINUTE, Integer.parseInt(delayMinute));
          }

          break;
        default:
          break;
      }
    }

    return calendar.getTime();
  }

  paint(g) {
    if (this._img == null) {
      this._img = Resources.IMG_AUTO_NODE;
    }

    // Call All Sub Elements PAINT METHOD.
    // 保存当前背景颜色...
    const old = this.bgcolor;
    // yx 判断Ctrl数组是否有值
    if(this._owner.checkSelectArr()){
      // yx 判断是否在ctrl数据中
      if(this._owner.isInZoom(this)){
        this.bgcolor = PaintElement.DEF_CURREDITCOLOR;
      }
    }else{
      if (this._owner.isCurrentToEdit(this)) {
        this.bgcolor = PaintElement.DEF_CURREDITCOLOR;
      }
  
      if (this._owner.isCurrentSelected(this)) {
        this.bgcolor = PaintElement.DEF_SELECTEDCOLOR;
      }
    }

    for (let i = 0; i < this._subelems.length; i++) {
      const te = this._subelems[i];
      if (te instanceof PaintElement) {
        const se = te;
        se.paint(g);
      }
    }

    // Fill background
    this.width = Node.WIDTH;
    this.m_width = Node.M_WIDTH;
    this.m_height = Node.M_HEIGHT;
    this.height = Node.HEIGHT;
    this.resize();
    if(this._owner.isCurrentToEdit(this) || this._owner.isInZoom(this)){
      g.setColor(this.bgcolor);
      g.fillRect(this.x - 3, this.y - 3, this.width + 6, this.height + 6);
      g.setColor("#000");
      g.drawOvalBorder(this.x, this.y, this.width, this.height, Node.PAD,this.bgcolor);
    }else{
      g.setColor("#000");
      g.drawOvalBorder(this.x, this.y, this.width, this.height, Node.PAD);
    }

    // Draw Image
    g.drawImage(this._img, this._imgrect.x, this._imgrect.y, this._imgrect.width, this._imgrect.height);

    // g.setColor(Resources.COLOR.lightGray);
    // g.fillRect(this._txtrect.x, this._txtrect.y, this._txtrect.width, 10 + this._txtrect.height);
    // g.setColor(Resources.COLOR.black);
    // g.strokeRect(this._txtrect.x, this._txtrect.y, this._txtrect.width, 10 + this._txtrect.height);

    if (this.name != null && this.name != "") {
      let nameWidthMax = g.stringWidth(this.name) > 86 ? 100 : g.stringWidth(this.name); // 86：即七个汉字
      const tx = this._txtrect.x + (this._txtrect.width - nameWidthMax) / 2;

      const ty = this._txtrect.y + 2 * this._txtrect.height;
      if (this._iscurrent) {
        // g.drawImage(Resources.IMG_CURRENT_NODE, this._txtrect.x,
        //     this._txtrect.y, this._txtrect.width, 10 + this._txtrect.height);
      } else {
        // g.drawImage(Resources.IMG_BACKGROUND_NODE,
        //     this._txtrect.x, this._txtrect.y, this._txtrect.width,
        // 		10 + this._txtrect.height);
      }
      g.setColor(Resources.COLOR.black);
      // g.drawString(StringUtil.dencodeHTML(this.name), tx + this.name.length, ty - 10);
      g.drawText(StringUtil.dencodeHTML(this.name), tx + 10, ty - 10, 80);
    }

    // 恢复当前背景颜色
    this.bgcolor = old;
  }

  toXML() {
    let str = "<cn.myapps.runtime.workflow.element.AutoNode>\n";

    str += "<isgather>" + this.isgather + "</isgather>\n";
    str += "<issplit>" + this.issplit + "</issplit>\n";
    str += "<autoAuditType>" + this.autoAuditType + "</autoAuditType>\n";
    str += "<autoAuditTimeEditMode>" + this.autoAuditTimeEditMode + "</autoAuditTimeEditMode>\n";
    str += "<auditDateTime>" + this.replaceCharacter(this.auditDateTime) + "</auditDateTime>\n";
    str += "<auditDateTimeScript><![CDATA[" + this.replaceXML(this.auditDateTimeScript) + " ]]></auditDateTimeScript>\n";
    str += "<delayTime>" + this.replaceCharacter(this.delayTime) + "</delayTime>\n";
    str += "<delayDay>" + this.replaceCharacter(this.delayDay) + "</delayDay>\n";
    str += "<delayHour>" + this.replaceCharacter(this.delayHour) + "</delayHour>\n";
    str += "<delayMinute>" + this.replaceCharacter(this.delayMinute) + "</delayMinute>\n";
    str += "<splitStartNode>" + this.replaceCharacter(this.splitStartNode) + "</splitStartNode>\n";
    str += super.toXML();

    str += "</cn.myapps.runtime.workflow.element.AutoNode>\n";
    return str;
  }
  //xh
  check() {
    let result = super.check();
    //节点状态标签和名称填写校验
    if (!result) {
      return false;
    }
    //聚合选项校验
    if (this.isgather) {
      if (!this.splitStartNode) {
        alert("请为" + this.name + "节点选择聚合审批节点(若选择聚合选项，则流程中至少要有一个节点开启并行)！");
        return false;
      }
    }
    //触发时机选项校验
    if (this.autoAuditTimeEditMode) {
      if (this.autoAuditTimeEditMode == 1) {
        if (this.autoAuditType) {
          if (this.autoAuditType == 2) {
            if (!this.auditDateTime) {
              alert("请为" + this.name + "节点填写指定触发时间(格式为:xxxx年xx月xx日 xx时:xx分)！");
              return false;
            }
          }

          if (this.autoAuditType == 3) {
            if (Number(this.delayMinute) == 0 && Number(this.delayDay) == 0 && Number(this.delayHour) == 0) {
              alert("请为" + this.name + "节点填写触发的滞后时间！");
              return false;
            }

            if (Number(this.delayMinute) < 0 || isNaN(Number(this.delayMinute)) || Number(this.delayDay) < 0 || isNaN(Number(this.delayDay)) || Number(this.delayHour) < 0 || isNaN(Number(this.delayHour))) {
              alert(this.name + "节点触发的滞后时间格式有误！");
              return false;
            }
          }
        } else {
          alert("请为" + this.name + "节点选择设计时触发的时机！");
          return false;
        }
      }

      if (this.autoAuditTimeEditMode == 2) {
        if (!this.auditDateTimeScript.trim()) {
          alert("请为" + this.name + "节点填写触发时的脚本！");
          return false;
        }
      }
    } else {
      alert("请为" + this.name + "节点选择设计或脚本中的任一模式！");
      return false;
    }

    return true;
  }
  //xh
  hasRelation(relation) {
    let { input, output } = super.hasRelation(relation);

    if (input == 0 && output == 0) {
      alert("请创建" + this.name + "节点的关联线！");
      return false;
    } else if (output > 0 && input == 0) {
      alert(this.name + "节点缺少接入节点！");
      return false;
    } else if (input > 0 && output == 0) {
      alert("请为" + this.name + "节点连接其他节点！");
      return false;
    } else {
      return true;
    }
  }

  // yx
  hasSomeRelation (relation) {
    return super.hasRelation(relation)
  }

  showTips(g) {
    this.drawTips(g, this.name);
  }

  errorCheck() {
    let errorStr = super.errorCheck();
    if (this.name != "") {
      if (this.isgather && this.splitStartNode == "") {
        errorStr += `${this.name} node.error.choosesplitstartnode` + `;`;
      }
    }
    return errorStr;
  }
}

/**
 * 马上审批
 */
AutoNode.AUTO_AUDIT_TYPE_IMMEDIATELY = 1;
/**
 * 指定时间审批
 */
AutoNode.AUTO_AUDIT_TYPE_SPECIFY = 2;
/**
 * 滞后一段时间审批
 */
AutoNode.AUTO_AUDIT_TYPE_DELAY = 3;

/**
 * 自动审批编辑模式-设计
 */
AutoNode.AUTO_AUDIT_EDIT_MODE_DESIGN = 1;

/**
 * 自动审批编辑模式-代码
 */
AutoNode.AUTO_AUDIT_EDIT_MODE_CODE = 2;

export default AutoNode;
