package com.bcxin.sync.task;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.entity.datasync.OrganizationOpenEntity;
import com.bcxin.sync.entity.tenant.TlkEventProjectEntity;
import com.bcxin.sync.entity.tenant.TlkEventTeamEntity;
import com.bcxin.sync.service.datasync.OrganizationOpenLogService;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import com.bcxin.sync.service.tenant.TlkEventProjectService;
import com.bcxin.sync.service.tenant.TlkEventTeamService;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * description：团队组织结束时间定时任务
 * author：linchunpeng
 * date：2025/1/10
 */
@Slf4j
@Component
public class TeamEndTimeTask {

    @Autowired
    private OrganizationOpenService organizationOpenService;
    @Autowired
    private OrganizationOpenLogService organizationOpenLogService;
    @Autowired
    private TlkEventProjectService tlkEventProjectService;
    @Autowired
    private TlkEventTeamService tlkEventTeamService;
    @Autowired
    private RedissonClient redissonClient;

    //每天凌晨1点执行
    @Scheduled(cron = "0 0 1 * * ?")
    public void teamEndTimeTask() {
        log.info("===================================团队组织结束时间定时任务===================================");
        //分布式锁key
        String lockKey = "REDISSON_LOCK_TEAM_END_TIME_TASK";
        //取锁
        log.info("团队组织结束时间定时任务， lockKey：{}，取锁中.....", lockKey);
        RLock lock = redissonClient.getLock(lockKey);
        try {
            //加锁，并设置过期时间 300s
            if (lock.tryLock(2, 600, TimeUnit.SECONDS)) {
                log.info("取到锁");
                List<TlkEventProjectEntity> projectList = tlkEventProjectService.getDayBeforeYesterdayEndProject();
                if (CollectionUtil.isNotEmpty(projectList)) {
                    List<String> orgIdList = new ArrayList<>();
                    List<String> projectIdList = new ArrayList<>();
                    for (TlkEventProjectEntity projectEntity : projectList) {
                        if (!tlkEventProjectService.isExistEndDateBeginYesterday(projectEntity.getItemDomainId())) {
                            //不存在结束时间大于等于昨天的的，就解除授权
                            orgIdList.add(projectEntity.getItemDomainId());
                            projectIdList.add(projectEntity.getId());
                        }
                    }
                    List<TlkEventTeamEntity> teamList = tlkEventTeamService.getTeamByProjectIdList(projectIdList);
                    if (CollectionUtil.isNotEmpty(teamList)) {
                        for (TlkEventTeamEntity teamEntity : teamList) {
                            if (!tlkEventTeamService.isExistEndDateBeginYesterday(teamEntity.getItemTeamId())) {
                                //不存在结束时间大于等于昨天的的，就解除授权
                                orgIdList.add(teamEntity.getItemTeamId());
                            }
                        }
                    }
                    Date now = new Date();
                    List<OrganizationOpenEntity> openList = organizationOpenService.getByOrganizationIdList(orgIdList, OpenType.TLJ.getCode());
                    if (CollectionUtil.isNotEmpty(openList)) {
                        for (OrganizationOpenEntity openEntity : openList) {
                            int openDay = (int) DateUtil.between(openEntity.getEndTime(), now, DateUnit.DAY, false);
                            openEntity.setEndTime(now);
                            //记录日志
                            organizationOpenLogService.createLog(openEntity.getOrganizationId(), openEntity.getOrganizationName(),
                                    openEntity.getOpenType(), openDay);
                        }
                        organizationOpenService.updateBatchById(openList);
                    }
                }
            } else {
                log.info("没有取到锁，跳过");
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("团队组织结束时间定时任务异常，{}", e.getMessage(), e);
        } finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
            log.info("团队组织结束时间定时任务，lockKey：{}，解锁", lockKey);
        }
        log.info("===================================团队组织结束时间定时任务结束===================================");
    }

}
