package com.bcxin.sync.service.tuoluojiang;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.utils.HttpUtil;
import com.bcxin.sync.common.utils.TuoluojiangUtil;
import com.bcxin.sync.configs.SyncConfig;
import com.bcxin.sync.dao.mapper.tuoluojiang.EbUserMapper;
import com.bcxin.sync.dtos.response.BcxEmployeeResponse;
import com.bcxin.sync.entity.tuoluojiang.EbAdminEntity;
import com.bcxin.sync.entity.tuoluojiang.EbAdminInfoEntity;
import com.bcxin.sync.entity.tuoluojiang.EbUserEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * description：陀螺匠-用户服务层
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class EbUserService extends ServiceImpl<EbUserMapper, EbUserEntity> {

    @Autowired
    private SyncConfig syncConfig;
    @Autowired
    private EbAdminService ebAdminService;
    @Autowired
    private EbAdminInfoService ebAdminInfoService;

    @Resource(name = "taskExecutor")
    private TaskExecutor taskExecutor;
    @Value("${spring.profiles.active}")
    private String activeFile;

    @Autowired
    private RedisCache redisCache;

    /**
     * description：批量创建用户
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public String userCreateBatch(List<BcxEmployeeResponse> employeeResponseList) {
        StringBuffer syncResult = new StringBuffer(256);
        log.info("批量创建用户，用户总数：{}", employeeResponseList.size());
        syncResult.append("\n批量创建用户，用户总数：").append(employeeResponseList.size());
        //根据分配大小，创建线程池大小，最多开5个线程
        CountDownLatch countDownLatch = new CountDownLatch(employeeResponseList.size());

        for (BcxEmployeeResponse response : employeeResponseList) {
            taskExecutor.execute(() -> {
                //执行同步用户数据
                try {
                    this.userCreate(response.getUserId(), response.getCredentialNumber(), response.getCardPhoto(),
                            response.getHeadPhoto(), response.getEmployeeName(), response.getTelephone());
                } catch (Exception e) {
                    log.error("同步用户数据异常，用户数据：{}，{}", JSONObject.toJSONString(response), e.getMessage(), e);
                    syncResult.append("\n同步用户数据异常，用户数据：").append(JSONObject.toJSONString(response)).append("，").append(e);
                }
                countDownLatch.countDown();
            });
        }

        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return syncResult.toString();
    }


    /**
     * description：创建用户
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public void userCreate(String userId, String cardNo, String cardPhoto, String headPhoto, String name, String mobile) {
        if (headPhoto == null) {
            headPhoto = "";
        }
        EbUserEntity entity = this.getUserByUserId(userId);
        if (entity == null) {
            long timeSecond = System.currentTimeMillis()/1000;
            Map<String, String> headerMap = TuoluojiangUtil.getHeaderMap("user/create".concat("POST") + timeSecond);

            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("uid", userId);
            paramMap.put("name", name);
            paramMap.put("card_id", cardNo);
            paramMap.put("phone", mobile);
            if (StringUtils.isNotBlank(cardPhoto)) {
                paramMap.put("avatar", cardPhoto);
            }
            paramMap.put("card_front", headPhoto);

            HttpUtil.post(syncConfig.getApiHost().getTuoluojiang().concat("/api/ent/internal/user/create"), JSONObject.toJSONString(paramMap), headerMap, "test".equals(activeFile));
        } else {
            boolean isUpdateCardNo = false;
            if (!cardNo.equals(entity.getCardId())) {
                isUpdateCardNo = true;
                entity.setCardId(cardNo);
            }
            entity.setRealName(name);
            entity.setPhone(mobile);
            if (StringUtils.isNotBlank(cardPhoto)) {
                entity.setAvatar(cardPhoto);
            }
            this.updateById(entity);
            //同步修改eb_user_resume
            this.baseMapper.updateUserHeadPhoto(entity.getUid(), headPhoto);
            //同步修改admin
            List<EbAdminEntity> ebAdminEntityList = ebAdminService.getListByUserId(entity.getUid());
            if (CollectionUtil.isNotEmpty(ebAdminEntityList)) {
                for (EbAdminEntity ebAdminEntity : ebAdminEntityList) {
                    ebAdminEntity.setName(name);
                    ebAdminEntity.setPhone(mobile);
                    if (StringUtils.isNotBlank(cardPhoto)) {
                        ebAdminEntity.setAvatar(cardPhoto);
                    }
                }
                ebAdminService.updateBatchById(ebAdminEntityList);
            }
            if (isUpdateCardNo) {
                String key = "data-sync:tenant-user:card-no:" + userId;
                redisCache.deleteObject(key);
                //同步修改adminInfo
                List<EbAdminInfoEntity> ebAdminInfoEntityList = ebAdminInfoService.getListByUserId(entity.getUid());
                if (CollectionUtil.isNotEmpty(ebAdminInfoEntityList)) {
                    for (EbAdminInfoEntity ebAdminInfoEntity : ebAdminInfoEntityList) {
                        ebAdminInfoEntity.setCardId(cardNo);
                    }
                    ebAdminInfoService.updateBatchById(ebAdminInfoEntityList);
                }
            }
        }
    }

    /**
     * description：根据智能人事用户id获取
     * author：linchunpeng
     * date：2024/12/31
     */
    public EbUserEntity getUserByUserId(String userId) {
        LambdaQueryChainWrapper<EbUserEntity> lqw = this.lambdaQuery();
        lqw.eq(EbUserEntity::getUid, userId);
        return lqw.one();
    }
}
