package com.bcxin.sync.service.tuoluojiang;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.emus.DataOperationType;
import com.bcxin.sync.common.utils.HttpUtil;
import com.bcxin.sync.common.utils.TuoluojiangUtil;
import com.bcxin.sync.configs.SyncConfig;
import com.bcxin.sync.dao.mapper.tuoluojiang.EbAdminMapper;
import com.bcxin.sync.dtos.kafka.message.EmployeeSyncMessage;
import com.bcxin.sync.dtos.kafka.message.ExternalMemberSyncMessage;
import com.bcxin.sync.dtos.response.BcxEmployeeResponse;
import com.bcxin.sync.entity.tenant.TenantDepartmentEmployeeRelationsEntity;
import com.bcxin.sync.entity.tuoluojiang.EbAdminEntity;
import com.bcxin.sync.entity.tuoluojiang.EbAdminInfoEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * description：陀螺匠-职员服务层
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class EbAdminService extends ServiceImpl<EbAdminMapper, EbAdminEntity> {

    @Autowired
    private EbFrameAssistService ebFrameAssistService;
    @Autowired
    private EbFrameService ebFrameService;
    @Autowired
    private EbEnterpriseService ebEnterpriseService;
    @Autowired
    private EbAdminInfoService ebAdminInfoService;
    @Autowired
    private EbAttendanceWhitelistService ebAttendanceWhitelistService;
    @Autowired
    private EbEnterpriseRoleUserService ebEnterpriseRoleUserService;
    @Autowired
    private SyncConfig syncConfig;

    @Resource(name = "taskExecutor")
    private TaskExecutor taskExecutor;
    @Value("${spring.profiles.active}")
    private String activeFile;

    @Autowired
    private RedisCache redisCache;

    /**
     * description：批量添加员工
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public String userJoinBatch(String orgId, Long entid, List<BcxEmployeeResponse> employeeResponseList, Long roleId,
                              Map<String, List<TenantDepartmentEmployeeRelationsEntity>> employeeDepartmentMap) {
        StringBuffer syncResult = new StringBuffer(256);
        log.info("批量添加员工，总数：{}", employeeResponseList.size());
        syncResult.append("\n批量添加员工，总数：").append(employeeResponseList.size());
        //根据分配大小，创建线程池大小，最多开5个线程
        CountDownLatch countDownLatch = new CountDownLatch(employeeResponseList.size());

        for (BcxEmployeeResponse response : employeeResponseList) {
            taskExecutor.execute(() -> {
                //执行添加员工
                try {
                    this.userJoin(orgId, entid, response.getUserId(), response.getId(), employeeDepartmentMap.get(response.getId()), null, null);
                    this.setAdminOtherInfo(response.getId(), roleId, response.getCardPhoto(), null, response.getIsDomainAdmin(), 0);
                } catch (Exception e) {
                    log.error("添加员工异常，员工数据：{}，{}", JSONObject.toJSONString(response), e.getMessage(), e);
                    syncResult.append("\n添加员工异常，员工数据：").append(JSONObject.toJSONString(response)).append("，").append(e);
                }
                countDownLatch.countDown();
            });
        }
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return syncResult.toString();
    }

    /**
     * description：添加员工
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public void userJoin(String orgId, Long entid, String userId, String employeeId,
                         List<TenantDepartmentEmployeeRelationsEntity> employeeDepartmentList,
                         EmployeeSyncMessage employee, ExternalMemberSyncMessage member ) {
        EbAdminEntity entity = this.getByEmployeeId(employeeId);
        if (entity == null) {
            log.info("职员不存在，调用接口接入企业");
            long timeSecond = System.currentTimeMillis()/1000;
            Map<String, String> headerMap = TuoluojiangUtil.getHeaderMap("user/join".concat("POST") + timeSecond);

            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("uid", userId);
            paramMap.put("ent_id", orgId);
            paramMap.put("uniqued", employeeId);

            List<Map<String, Object>> frameList = new ArrayList<>();
            Set<String> employeeIdSet = new HashSet<>();
            if (CollectionUtil.isNotEmpty(employeeDepartmentList)) {
                for (TenantDepartmentEmployeeRelationsEntity relationsEntity : employeeDepartmentList) {
                    Map<String, Object> map = new HashMap<>();
                    map.put("admin", relationsEntity.getLeaderType());
                    if (employeeIdSet.add(relationsEntity.getEmployeeId())) {
                        map.put("master", 1);
                    }
                    map.put("frame_id", relationsEntity.getDepartmentId());
                    map.put("uniqued", relationsEntity.getId());
                    frameList.add(map);
                }
            } else {
                Map<String, Object> map = new HashMap<>();
                map.put("admin", 0);
                map.put("master", 1);
                map.put("frame_id", ebFrameService.getTopUniquedFrameId(entid));
                map.put("uniqued", employeeId.concat("_").concat(orgId));
                frameList.add(map);
            }
            paramMap.put("frames", frameList);
            HttpUtil.post(syncConfig.getApiHost().getTuoluojiang().concat("/api/ent/internal/user/join"), JSONObject.toJSONString(paramMap), headerMap, "test".equals(activeFile));
        } else {
            log.info("职员已存在，更新");
            boolean isLeave = false;
            if (employee != null) {
                if (employee.getBeforeStatus().getCode() != employee.getAfterStatus().getCode()) {
                    EbAdminInfoEntity adminInfoEntity = ebAdminInfoService.getByUserIdAndEntid(entity.getUid(), entity.getEntid());
                    //在离职状态变化
                    if (employee.getAfterStatus().getCode() == 1) {
                        //离职
                        adminInfoEntity.setType(4);
                        adminInfoEntity.setQuitTime(DateUtil.formatDateTime(employee.getLeaveDate()));
                        isLeave = true;
                    } else if (employee.getBeforeStatus().getCode() == 1 && employee.getAfterStatus().getCode() == 0) {
                        //复职
                        adminInfoEntity.setType(1);
                        adminInfoEntity.setWorkTime(DateUtil.formatDateTime(employee.getHireDate()));
                    }
                    ebAdminInfoService.updateById(adminInfoEntity);
                }
            }
            if (member != null && member.getOpType().getCode() == DataOperationType.UPDATE.getCode()) {
                if (member.getBeforeApprovedInformationStatus().intValue() != member.getAfterApprovedInformationStatus().intValue()) {
                    EbAdminInfoEntity adminInfoEntity = ebAdminInfoService.getByUserIdAndEntid(entity.getUid(), entity.getEntid());
                    //在离职状态变化
                    if (member.getAfterApprovedInformationStatus() == 2 || member.getAfterApprovedInformationStatus() == 3) {
                        //变更后是审批不通过或者删除，属于：离职
                        adminInfoEntity.setType(4);
                        adminInfoEntity.setQuitTime(DateUtil.formatDateTime(member.getDeletedTime()));
                        isLeave = true;
                    } else if (member.getAfterApprovedInformationStatus() == 1) {
                        //变更后是审批通过，属于：复职
                        adminInfoEntity.setType(1);
                        adminInfoEntity.setWorkTime(DateUtil.formatDateTime(member.getJoinTime()));
                    }
                    ebAdminInfoService.updateById(adminInfoEntity);
                }
            }
            //删除部门人员关联
            ebFrameAssistService.frameAssistDelete(entid, entity.getId());
            if (!isLeave) {
                //在职，才需要同步部门
                if (CollectionUtil.isNotEmpty(employeeDepartmentList)) {
                    int isMastart = 1;
                    for (TenantDepartmentEmployeeRelationsEntity relationsEntity : employeeDepartmentList) {
                        ebFrameAssistService.frameAssistCreate(entid, relationsEntity.getDepartmentId(), entity.getId(),
                                relationsEntity.getLeaderType(), isMastart, relationsEntity.getId());
                        isMastart = 0;
                    }
                } else {
                    ebFrameAssistService.frameAssistCreate(entid, ebFrameService.getTopUniquedFrameId(entid), entity.getId(),
                            0, 1, employeeId.concat("_").concat(orgId));
                }
            }
        }
    }

    /**
     * description：设置员工其他信息
     * author：linchunpeng
     * date：2025/3/13
     */
    @Transactional
    public EbAdminEntity setAdminOtherInfo(String employeeId, Long roleId, String cardPhoto, EmployeeSyncMessage employee,
                                  Integer isAdmin, Integer isPart) {
        log.info("设置员工其他信息");
        EbAdminEntity entity = this.getByEmployeeId(employeeId);
        if (entity != null) {
            entity.setIsAdmin(isAdmin);
            if (StringUtils.isNotBlank(cardPhoto)) {
                entity.setAvatar(cardPhoto);
            }
            log.info("设置用户角色（OA与考勤）");
            String roles = entity.getRoles();
            if (StringUtils.isBlank(roles) || roles.length()<3) {
                roles = "["+ roleId.toString()+"]";
            } else {
                if (Arrays.stream(roles.substring(1, roles.length() - 1).split(",")).noneMatch(role -> role.equals(roleId.toString()))) {
                    roles = roles.substring(0, roles.length() - 1).concat(",").concat(roleId.toString()).concat("]");
                }
            }
            entity.setRoles(roles);
            this.updateById(entity);

            if (isPart != null && isPart == 1) {
                EbAdminInfoEntity adminInfoEntity = ebAdminInfoService.getByUserIdAndEntid(entity.getUid(), entity.getEntid());
                adminInfoEntity.setIsPart(isPart);
                ebAdminInfoService.updateById(adminInfoEntity);
            }

            ebEnterpriseRoleUserService.roleUserCreateAndSave(entity.getEntid(), roleId, entity.getId());
            if (isAdmin == 1) {
                log.info("组织管理员，同步创建考勤组白名单");
                ebAttendanceWhitelistService.attendanceWhitelistCreate(entity.getEntid(), entity.getId(), 1);
            } else if (employee != null && employee.getBeforeIsDomainAdmin() != null && employee.getBeforeIsDomainAdmin() == 1
                    && employee.getAfterIsDomainAdmin() != null && employee.getAfterIsDomainAdmin() == 0){
                log.info("移除组织管理员，同步删除考勤组白名单");
                ebAttendanceWhitelistService.attendanceWhitelistDelete(entity.getId(), 1);
            }
            return entity;
        }
        return null;
    }

    /**
     * description：修改组织管理员的信息
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public EbAdminEntity getEntAdmin(Long entid) {
        LambdaQueryChainWrapper<EbAdminEntity> lqw = this.lambdaQuery();
        lqw.eq(EbAdminEntity::getEntid, entid);
        lqw.eq(EbAdminEntity::getIsAdmin, 1);
        List<EbAdminEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }

    /**
     * description：根据智能人事职员id获取陀螺匠职员Id
     * author：linchunpeng
     * date：2024/12/31
     */
    public Long getIdByEmployeeId(String employeeId, Long entid) {
        String key = "data-sync:employee:admin-id:" + employeeId;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString());
        }
        LambdaQueryChainWrapper<EbAdminEntity> lqw = this.lambdaQuery();
        lqw.eq(EbAdminEntity::getEntid, entid);
        lqw.eq(EbAdminEntity::getUniqued, employeeId);
        EbAdminEntity adminEntity = lqw.one();
        if (adminEntity != null) {
            redisCache.setCacheObject(key, adminEntity.getId(), 1, TimeUnit.DAYS);
            return adminEntity.getId();
        }
        return null;
    }

    /**
     * description：根据智能人事职员id获取陀螺匠职员
     * author：linchunpeng
     * date：2024/12/31
     */
    public EbAdminEntity getByEmployeeId(String employeeId) {
        LambdaQueryChainWrapper<EbAdminEntity> lqw = this.lambdaQuery();
        lqw.eq(EbAdminEntity::getUniqued, employeeId);
        return lqw.one();
    }

    /**
     * description：根据用户id获取陀螺匠职员列表
     * author：linchunpeng
     * date：2024/12/31
     */
    public List<EbAdminEntity> getListByUserId(String userId) {
        LambdaQueryChainWrapper<EbAdminEntity> lqw = this.lambdaQuery();
        lqw.eq(EbAdminEntity::getUid, userId);
        return lqw.list();
    }

    /**
     * description：根据职员id列表获取陀螺匠职员列表
     * author：linchunpeng
     * date：2024/12/31
     */
    public List<EbAdminEntity> getListByEmployeeIdList(List<String> employeeIdList, Long entid) {
        LambdaQueryChainWrapper<EbAdminEntity> lqw = this.lambdaQuery();
        lqw.in(EbAdminEntity::getUniqued, employeeIdList);
        return lqw.list();
    }
}