package com.bcxin.sync.service.tenant;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.utils.IdGeneratorSnowflake;
import com.bcxin.sync.dao.mapper.tenant.RbacCustomRoleUserMapper;
import com.bcxin.sync.entity.tenant.RbacCustomRoleUserEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;

/**
 * description：智能人事-角色用户关联表
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class RbacCustomRoleUserService extends ServiceImpl<RbacCustomRoleUserMapper, RbacCustomRoleUserEntity> {

    @Autowired
    private IdGeneratorSnowflake snowflake;

    /**
     * description：通过角色id，查询角色用户关联表
     * author：linchunpeng
     * date：2025/2/19
     */
    public List<RbacCustomRoleUserEntity> getListByRbacRoleId(Long rbacRoleId) {
        LambdaQueryChainWrapper<RbacCustomRoleUserEntity> lqw = this.lambdaQuery();
        lqw.eq(RbacCustomRoleUserEntity::getRbacRoleId, rbacRoleId);
        return lqw.list();
    }

    /**
     * description：判断职员和角色是否关联
     * author：linchunpeng
     * date：2025/2/26
     */
    public boolean isExistEmployee(Long rbacRoleId, String tenantEmployeeId) {
        LambdaQueryChainWrapper<RbacCustomRoleUserEntity> lqw = this.lambdaQuery();
        lqw.eq(RbacCustomRoleUserEntity::getRbacRoleId, rbacRoleId);
        lqw.eq(RbacCustomRoleUserEntity::getTenantEmployeeId, tenantEmployeeId);
        return lqw.exists();
    }

    /**
     * description：创建职员和角色关联
     * author：linchunpeng
     * date：2025/2/28
     */
    public RbacCustomRoleUserEntity create(String organizationId, Long roleId, String tenantEmployeeId) {
        RbacCustomRoleUserEntity roleUserEntity = new RbacCustomRoleUserEntity();
        roleUserEntity.setId(snowflake.snowflakeId());
        roleUserEntity.setOrganizationId(organizationId);
        roleUserEntity.setRbacRoleId(roleId);
        roleUserEntity.setTenantEmployeeId(tenantEmployeeId);
        roleUserEntity.setCreateTime(new Date());
        return roleUserEntity;
    }

    /**
     * description：创建职员和角色关联，并保存
     * author：linchunpeng
     * date：2025/2/28
     */
    @Transactional
    public void createAndSave(String organizationId, Long roleId, String tenantEmployeeId) {
        if (!this.isExistEmployee(roleId, tenantEmployeeId)) {
            RbacCustomRoleUserEntity roleUserEntity = this.create(organizationId, roleId, tenantEmployeeId);
            this.save(roleUserEntity);
        }
    }

    /**
     * description：删除职员和角色关联，并保存
     * author：linchunpeng
     * date：2025/2/28
     */
    @Transactional
    public void deleteByRoleIdAndEmployeeId(Long roleId, String tenantEmployeeId) {
        LambdaQueryChainWrapper<RbacCustomRoleUserEntity> lqw = this.lambdaQuery();
        lqw.eq(RbacCustomRoleUserEntity::getRbacRoleId, roleId);
        lqw.eq(RbacCustomRoleUserEntity::getTenantEmployeeId, tenantEmployeeId);
        List<RbacCustomRoleUserEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            this.removeById(list.get(0).getId());
        }
    }
}