package com.bcxin.sync.consumer;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.sync.common.CommonConstant;
import com.bcxin.sync.common.emus.DataOperationType;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.common.utils.KafkaMessageUtil;
import com.bcxin.sync.dtos.kafka.message.TenantUserSyncMessage;
import com.bcxin.sync.dtos.response.BcxEmployeeResponse;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import com.bcxin.sync.service.jvs.SysUserInfoService;
import com.bcxin.sync.service.tenant.TenantEmployeesService;
import com.bcxin.sync.service.tuoluojiang.EbUserService;
import com.bcxin.sync.service.yehuo.TUserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户kafka消息监听器
 */
@Slf4j
@Component
public class TenantUsersKafkaConsumer {

    @Autowired
    private EbUserService ebUserService;
    @Autowired
    private SysUserInfoService sysUserInfoService;
    @Autowired
    private TenantEmployeesService tenantEmployeesService;
    @Autowired
    private OrganizationOpenService organizationOpenService;
    @Autowired
    private TUserService tUserService;

    @KafkaListener(id = "bcx_data_sync_listener_tenant_users_to_tlj",
            topics = CommonConstant.TENANT_USERS_KAFKA_TOPIC,
            groupId = "bcx_data_sync_tenant_users_to_tlj")
    public void bcxDataSyncTenantUsersToTlj(ConsumerRecord<String, String> record, Acknowledgment ack) {
        TenantUserSyncMessage user = KafkaMessageUtil.transferKafkaUserRecord2Message(record);
        log.info("同步陀螺匠，kafka监听tenant-users表，数据转化为：{}", JSONObject.toJSONString(user));
        try {
            if (user != null) {
                List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByUserIdList(Collections.singletonList(user.getUserId()), 0);
                if (CollectionUtil.isNotEmpty(employeeResponseList)) {
                    log.info("同步陀螺匠，有职员信息，数量：{}", employeeResponseList.size());
                    List<String> orgIdList = employeeResponseList.stream().map(BcxEmployeeResponse::getOrganizationId).collect(Collectors.toList());
                    boolean openAndEffective = organizationOpenService.isOpenAndEffective(orgIdList, OpenType.TLJ.getCode());
                    if (openAndEffective) {
                        //有开通陀螺匠，才需要同步用户信息
                        log.info("同步陀螺匠，组织id列表：{}中，有开通陀螺匠，所以需要同步用户信息", JSONObject.toJSONString(orgIdList));
                        ebUserService.userCreate(user.getUserId(), employeeResponseList.get(0).getCredentialNumber(),
                                employeeResponseList.get(0).getCardPhoto(), employeeResponseList.get(0).getHeadPhoto(),
                                user.getAfterName(), user.getAfterTelephone());
                    } else {
                        log.info("同步陀螺匠，组织id列表：{}中，都没有开通陀螺匠，所以不需要同步用户信息", JSONObject.toJSONString(orgIdList));
                    }
                }
            }
        } catch (Exception e) {
            log.error("同步陀螺匠，kafka监听tenant-users表，异常：{}", e.getMessage(), e);
        }
        ack.acknowledge();
    }

    @KafkaListener(id = "bcx_data_sync_listener_tenant_users_to_jvs",
            topics = CommonConstant.TENANT_USERS_KAFKA_TOPIC,
            groupId = "bcx_data_sync_tenant_users_to_jvs")
    public void bcxDataSyncTenantUsersToJvs(ConsumerRecord<String, String> record, Acknowledgment ack) {
        log.info("同步JVS，kafka监听tenant-users表，数据：{}", record.value());
        TenantUserSyncMessage user = KafkaMessageUtil.transferKafkaUserRecord2Message(record);
        log.info("同步JVS，kafka监听tenant-users表，数据转化为：{}", JSONObject.toJSONString(user));
        try {
            List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByUserIdList(Collections.singletonList(user.getUserId()), null);
            if (CollectionUtil.isNotEmpty(employeeResponseList)) {
                log.info("同步JVS，有职员信息，数量：{}", employeeResponseList.size());
                List<String> orgIdList = employeeResponseList.stream().map(BcxEmployeeResponse::getOrganizationId).collect(Collectors.toList());
                boolean openAndEffective = organizationOpenService.isOpenAndEffective(orgIdList, OpenType.JVS.getCode());
                if (openAndEffective) {
                    //有开通JVS，才需要同步用户信息
                    log.info("同步JVS，组织id列表：{}中，有开通JVS，所以需要同步用户信息", JSONObject.toJSONString(orgIdList));
                    sysUserInfoService.create(user.getUserId(), employeeResponseList.get(0).getCredentialNumber(), user.getAfterName(), user.getAfterTelephone());
                } else {
                    log.info("同步JVS，组织id列表：{}中，都没有开通JVS，所以不需要同步用户信息", JSONObject.toJSONString(orgIdList));
                }
            }
        } catch (Exception e) {
            log.error("同步JVS，kafka监听tenant-users表，异常：{}", e.getMessage(), e);
        }
        ack.acknowledge();
    }

    @KafkaListener(id = "bcx_data_sync_listener_tenant_users_to_yehuo",
            topics = CommonConstant.TENANT_USERS_KAFKA_TOPIC,
            groupId = "bcx_data_sync_tenant_users_to_yehuo")
    public void bcxDataSyncTenantUsersToYehuo(ConsumerRecord<String, String> record, Acknowledgment ack) {
        log.info("同步野火，kafka监听tenant-users表，数据：{}", record.value());
        TenantUserSyncMessage user = KafkaMessageUtil.transferKafkaUserRecord2Message(record);
        log.info("同步野火，kafka监听tenant-users表，数据转化为：{}", JSONObject.toJSONString(user));
        try {
            if (user != null && user.getOpType().getCode() == DataOperationType.CREATE.getCode()) {
                //野火只需要同步新增
                log.info("同步野火，有新建用户，需要同步");
                tUserService.create(user.getUserId(), user.getAfterName(), user.getAfterTelephone());
            }
        } catch (Exception e) {
            log.error("同步野火，kafka监听tenant-users表，异常：{}", e.getMessage(), e);
        }
        ack.acknowledge();
    }
}
