package com.bcxin.sync.common.utils;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.sync.common.exception.GetResponseException;
import com.bcxin.sync.dtos.ivm.DeviceDto;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.bytedeco.ffmpeg.global.avcodec;
import org.bytedeco.ffmpeg.global.avutil;
import org.bytedeco.javacv.FFmpegFrameGrabber;
import org.bytedeco.javacv.FFmpegFrameRecorder;
import org.bytedeco.javacv.Frame;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Slf4j
public class HuaweiIvmUtil {

    /**
     * description：获取华为云IVM账户的AccessToken
     * author：linchunpeng
     * date：2025/5/27
     */
    public static String getAccessToken(String host, String userId, String ak, String sk, RedisCache redisCache) {
        log.info("获取华为云IVM账户的token");
        String key = "data-sync:huawei:ivm:access-token:" + userId;
        if (redisCache != null) {
            Object value = redisCache.getCacheObject(key);
            if (value != null) {
                return value.toString();
            }
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("ak", ak);
        paramMap.put("sk", sk);
        paramMap.put("force_update", true);

        String result = HttpUtil.post(host.concat("/v2/").concat(userId).concat("/enterprises/access-token"), JSONObject.toJSONString(paramMap), null, false);
        log.info("获取华为云IVM账户的token，结果：{}", result);
        if (result.equals("fail")) {
            //获取失败
            throw new GetResponseException(500, "登录失败，fail");
        }
        JSONObject resultObj = JSONObject.parseObject(result);
        if (resultObj.containsKey("access_token")) {
            //获取成功
            String token = resultObj.getString("access_token");
            if (redisCache != null) {
                redisCache.setCacheObject(key, token, 30, TimeUnit.MINUTES);
            }
            return token;
        } else {
            //获取失败
            throw new GetResponseException(500, "登录失败，" + resultObj.getString("msg"));
        }
    }


    /**
     * description：获取华为云IVM设备列表
     * author：linchunpeng
     * date：2025/5/27
     */
    public static List<DeviceDto> getDeviceList(String host, String userId, String accessToken) {
        log.info("获取华为云IVM设备列表");
        List<DeviceDto> resultList = new ArrayList<>();

        String result = HttpUtil.get(host.concat("/v1/").concat(userId).concat("/devices?limit=1000&offset=0"),
                getHeaderMap(accessToken), false);
        log.info("获取华为云IVM设备列表，结果：{}", result);
        if (!result.equals("fail")) {
            JSONObject resultObj = JSONObject.parseObject(result);
            if (resultObj.containsKey("total") && resultObj.getInteger("total") > 0) {
                //获取成功
                int total = resultObj.getInteger("total");
                resultList.addAll(resultObj.getJSONArray("devices").toJavaList(DeviceDto.class));
                if (total > 1000) {
                    //超过1页，需要分页获取
                    int page = total/1000;
                    for (int i = 1; i <= page; i++) {
                        String result_offset = HttpUtil.get(host.concat("/v1/").concat(userId).concat("/devices?limit=1000&offset=").concat(""+i)
                                , getHeaderMap(accessToken), false);
                        if (!result_offset.equals("fail")) {
                            JSONObject resultObj_offset = JSONObject.parseObject(result_offset);
                            if (resultObj_offset.containsKey("total") && resultObj_offset.getInteger("total") > 0) {
                                resultList.addAll(resultObj_offset.getJSONArray("devices").toJavaList(DeviceDto.class));
                            }
                        }
                    }
                }
            }
        }
        log.info("获取华为云IVM设备列表，结果对象：{}", JSONObject.toJSONString(resultList));
        return resultList;
    }


//    /**
//     * description：根据设备id，获取华为云IVM通道信息
//     * author：linchunpeng
//     * date：2025/5/27
//     */
//    public static ChannelDto getChannelByDeviceId(String host, String userId, String accessToken, String deviceId) {
//        log.info("根据设备id，获取华为云IVM通道信息，设备id：{}", deviceId);
//
//        String result = HttpUtil.get(host.concat("/v1/").concat(userId).concat("/channels?device_id=").concat(deviceId),
//                getHeaderMap(accessToken), false);
//        log.info("根据设备id，获取华为云IVM通道信息，设备id：{}，结果：{}", deviceId, result);
//        if (!result.equals("fail")) {
//            JSONObject resultObj = JSONObject.parseObject(result);
//            if (resultObj.containsKey("total") && resultObj.getInteger("total") > 0) {
//                //获取成功
//                List<ChannelDto> channelDtoList = resultObj.getJSONArray("channels").toJavaList(ChannelDto.class);
//                log.info("根据设备id，获取华为云IVM通道信息，设备id：{}，结果对象：{}", deviceId, JSONObject.toJSONString(channelDtoList));
//                return channelDtoList.get(0);
//            }
//        }
//        return new ChannelDto();
//    }

    /**
     * description：根据设备id，通道id，批量删除路线计划
     * author：linchunpeng
     * date：2025/5/27
     */
    public static boolean deleteDeviceRecordPlan(String host, String userId, String accessToken, String deviceId, String channelId) {
        log.info("根据设备id，通道id，批量删除路线计划，设备id：{}，通道id：{}", deviceId, channelId);
        Map<String, Object> paramMap = new HashMap<>();
        List<Map<String, Object>> channelList = new ArrayList<>();
        Map<String, Object> channel = new HashMap<>();
        channel.put("device_id", deviceId);
        channel.put("channel_id", channelId);
        channelList.add(channel);
        paramMap.put("channels", channelList);

        String result = HttpUtil.post(host.concat("/v1/").concat(userId).concat("/devices/channels/record-plan"),
                JSONObject.toJSONString(paramMap), getHeaderMap(accessToken), false);
        log.info("根据设备id：{}，通道id：{}，批量删除路线计划，结果：{}", deviceId, channelId, result);
        if (!result.equals("fail")) {
            JSONObject resultObj = JSONObject.parseObject(result);
            if (resultObj.containsKey("failed_num") && resultObj.getInteger("failed_num") == 0) {
                //删除成功
                return true;
            }
        }
        return false;
    }


    /**
     * description：根据设备id，设置通道录像计划
     * author：linchunpeng
     * date：2025/5/27
     */
    public static void setDeviceRecordPlan(String host, String userId, String accessToken, String deviceId, List<Date> startTimeList, List<Date> endTimeList) {
        log.info("根据设备id，设置通道录像计划，设备id：{}，开始时间：{}，结束时间：{}", deviceId, startTimeList, endTimeList);
//        ChannelDto channelDto = getChannelByDeviceId(host, userId, accessToken, deviceId);
//        String channelId = channelDto.getChannel_id();
        String channelId = deviceId;
        log.info("根据设备id：{}，获取通道id：{}", deviceId, channelId);
        if (StringUtils.isNotBlank(channelId)) {
            boolean deleteSuccess = deleteDeviceRecordPlan(host, userId, accessToken, deviceId, channelId);
            log.info("根据设备id：{}，通道id：{}，批量删除路线计划，结果：{}", deviceId, channelId, deleteSuccess);

            if (CollectionUtil.isNotEmpty(startTimeList)
                    && CollectionUtil.isNotEmpty(endTimeList)
                    && startTimeList.size() == endTimeList.size()) {
                log.info("根据设备id：{}，获取通道id：{}，计划录像时间不为空", deviceId, channelId);
                for (int i = 1; i < startTimeList.size(); i++) {
                    Date nextStartTime = startTimeList.get(i);
                    Date endTime = endTimeList.get(i-1);
                    if (nextStartTime.before(endTime)) {
                        nextStartTime = endTime;
                        startTimeList.set(i, nextStartTime);
                    }
                }
                for (int i = startTimeList.size() - 1; i >= 0; i--) {
                    Date startTime = startTimeList.get(i);
                    Date endTime = endTimeList.get(i);
                    if (endTime.before(startTime)) {
                        startTimeList.remove(i);
                        endTimeList.remove(i);
                    }
                }

                Map<String, Object> paramMap = new HashMap<>();
                List<Map<String, Object>> planList = new ArrayList<>();
                Map<String, Object> plan = new HashMap<>();
                plan.put("device_id", deviceId);
                plan.put("channel_id", channelId);
                plan.put("all_day", false);

                List<Map<String, Object>> timeSectionList = new ArrayList<>();
                for (int i = 0; i <startTimeList.size(); i++) {
                    Date startTime = startTimeList.get(i);
                    Date endTime = endTimeList.get(i);
                    if (startTime.after(endTime)) {
                        throw new GetResponseException(500, String.format("根据设备id，设置通道录像计划，设备id：%s，开始时间：%s，结束时间：%s，结束时间不能小于开始时间", deviceId, startTimeList, endTimeList));
                    }
                    int startDayOfWeek = DateUtil.getWeekDayOfDate(startTime);
                    int endDayOfWeek = DateUtil.getWeekDayOfDate(endTime);

                    String startStr = cn.hutool.core.date.DateUtil.format(startTime, "HH:mm:ss");
                    String endStr = cn.hutool.core.date.DateUtil.format(endTime, "HH:mm:ss");

                    if (startDayOfWeek == endDayOfWeek) {
                        //开始和结束在同一天
                        Map<String, Object> timeSection = new HashMap<>();
                        timeSection.put("day_of_week", startDayOfWeek);
                        timeSection.put("start_time", startStr);
                        timeSection.put("end_time", endStr);
                        timeSectionList.add(timeSection);
                    } else {
                        //开始和结束跨天
                        for (int d = startDayOfWeek; d <= endDayOfWeek; d++) {
                            Map<String, Object> timeSection = new HashMap<>();
                            timeSection.put("day_of_week", d);
                            if (d == startDayOfWeek) {
                                //第一天
                                timeSection.put("start_time", startStr);
                                timeSection.put("end_time", "23:59:59");
                            } else if (d == endDayOfWeek) {
                                //最后一天
                                timeSection.put("start_time", "00:00:00");
                                timeSection.put("end_time", endStr);
                            } else {
                                //中间天
                                timeSection.put("start_time", "00:00:00");
                                timeSection.put("end_time", "23:59:59");
                            }
                            timeSectionList.add(timeSection);
                        }
                    }
                }

                plan.put("time_sections", timeSectionList);
                planList.add(plan);
                paramMap.put("plans", planList);
                log.info("根据设备id，设置通道录像计划，设备id：{}，参数plans：{}", deviceId, planList);
                String result = HttpUtil.put(host.concat("/v1/").concat(userId).concat("/devices/channels/record-plan"),
                        JSONObject.toJSONString(paramMap), getHeaderMap(accessToken), false);
                log.info("根据设备id，设置通道录像计划，设备id：{}，结果：{}", deviceId, result);
                if (!result.equals("fail")) {
                    JSONObject resultObj = JSONObject.parseObject(result);
                    if (resultObj.containsKey("failed_num") && resultObj.getInteger("failed_num") == 0) {
                        //设置成功
                        log.info("根据设备id，设置通道录像计划，设备id：{}，设置成功", deviceId);
                    }
                } else {
                    log.info("根据设备id，设置通道录像计划，设备id：{}，设置失败", deviceId);
                }
            } else {
                log.info("根据设备id：{}，获取通道id：{}，计划录像时间为空", deviceId, channelId);
            }
        }
    }

    /**
     * description：根据设备id，获取华为云IVM回放地址
     * author：linchunpeng
     * date：2025/5/27
     */
    public static String getPlaybackByDeviceId(String host, String userId, String accessToken, String deviceId, String channelId,
                                                   String startTime, String endTime) {
        log.info("根据设备id，获取华为云IVM回放地址，设备id：{}，开始时间：{}，结束时间：{}", deviceId, startTime, endTime);
        String url = host.concat("/v2/").concat(userId).concat("/devices/").concat(deviceId)
                .concat("/channels/").concat(channelId).concat("/media/playback-connections?playback_protocol=HTTP_HLS&start_time=")
                .concat(startTime.replace(" ", "%20")).concat("&end_time=").concat(endTime.replace(" ", "%20")).concat("&record_position=CLOUD&expire_time=86400&use_times=1");
        String result = HttpUtil.get(url, getHeaderMap(accessToken), false);
        log.info("根据设备id，获取华为云IVM回放地址，设备id：{}，开始时间：{}，结束时间：{}，结果：{}", deviceId, startTime, endTime, result);
        String playUrl = "";
        if (!result.equals("fail")) {
            JSONObject resultObj = JSONObject.parseObject(result);
            if (resultObj.containsKey("playback_connections") ) {
                JSONObject playback_connections = resultObj.getJSONObject("playback_connections");
                if (playback_connections.containsKey("cloud_trans_connections")) {
                    JSONObject cloud_trans_connections = playback_connections.getJSONObject("cloud_trans_connections");
                    playUrl = getByUnicode(cloud_trans_connections.getString("playback_url"));
                }
            }
           log.info("回放地址：{}", playUrl);
            return playUrl;
        }
        return null;
    }


    /**
     * description：获取调用接口的头部参数与值
     * author：linchunpeng
     * date：2025/5/27
     */
    public static Map<String, String> getHeaderMap(String accessToken) {
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("Access-Token", accessToken);
        return headerMap;
    }

    /**
     * description：unicode解码
     * author：linchunpeng
     * date：2025/6/11
     */
    private static String getByUnicode(String unicodeStr) {
        Pattern pattern = Pattern.compile("\\\\u([0-9a-fA-F]{4})"); // 注意双斜杠表示单个斜杠字符
        Matcher matcher = pattern.matcher(unicodeStr);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            String hexStr = matcher.group(1); // 获取4位十六进制数
            char c = (char) Integer.parseInt(hexStr, 16); // 转换为字符
            matcher.appendReplacement(sb, String.valueOf(c)); // 替换为字符
        }
        matcher.appendTail(sb); // 添加剩余部分
        return sb.toString();
    }

    /**
     * description：下载视频流，并保存
     * author：linchunpeng
     * date：2025/6/11
     */
    public static boolean convertStreamToMp4(String inputUrl, String outputPath) {
        boolean result = true;
        FFmpegFrameGrabber grabber = null;
        FFmpegFrameRecorder recorder = null;
        try {
            // 1. 初始化抓取器（确保先创建抓取器）
            grabber = new FFmpegFrameGrabber(inputUrl);

            // 设置网络参数（使用TCP传输）
            grabber.setOption("rtsp_transport", "tcp");
            grabber.setOption("stimeout", "5000000"); // 5秒超时(单位微秒)
//            grabber.setFrameGrabTimeout(10_000); // JavaCV超时设置

            log.info("尝试连接源流: " + inputUrl);
            grabber.start();
            log.info("成功连接源流");

            // 2. 准备记录器配置
            int width = grabber.getImageWidth();
            int height = grabber.getImageHeight();
            int audioChannels = grabber.getAudioChannels();
            double frameRate = grabber.getFrameRate();

            log.info("视频信息: " + width + "x" + height + "@" + frameRate + "fps");
            log.info("音频通道: " + audioChannels);

            // 3. 创建记录器
            recorder = new FFmpegFrameRecorder(outputPath, width, height, audioChannels);

            // 4. 设置MP4格式的关键参数
            recorder.setFormat("mp4"); // 必须第一个设置

            // 解决时间戳问题
            recorder.setVideoOption("async", "1000");
            recorder.setVideoOption("vsync", "0"); // 禁用内部时间戳处理
            recorder.setVideoOption("use_wallclock_as_timestamps", "1");//使用系统时间作为基准
            recorder.setVideoOption("avoid_negative_ts", "make_zero");

            // 视频编码配置
            recorder.setVideoCodec(avcodec.AV_CODEC_ID_H264);
            recorder.setVideoOption("movflags", "+faststart"); // MP4快速启动
            recorder.setVideoOption("crf", "23"); // 质量参数
            recorder.setVideoOption("preset", "fast"); // 编码速度
            recorder.setFrameRate(frameRate > 0 ? frameRate : 30); // 默认30fps
            recorder.setGopSize((int) (frameRate * 2)); // 关键帧间隔（2秒）

            // 使用兼容性最好的像素格式
            recorder.setPixelFormat(avutil.AV_PIX_FMT_YUV420P);

            // 4. 设置比特率
            if (grabber.getVideoBitrate() > 0) {
                recorder.setVideoBitrate(grabber.getVideoBitrate());
            } else {
                // 估算比特率
                int bitrate = grabber.getImageWidth() * grabber.getImageHeight() * 5;
                recorder.setVideoBitrate(Math.min(bitrate, 2000000)); // 限制最大2Mbps
            }

            // 音频编码配置
            if (audioChannels > 0) {
                recorder.setAudioCodec(avcodec.AV_CODEC_ID_AAC);
                recorder.setAudioChannels(audioChannels);
                recorder.setAudioBitrate(64000); // 64 kbps
                // 确保采样率有效
                int sampleRate = grabber.getSampleRate();
                if (sampleRate <= 0) {
                    sampleRate = 44100; // 默认44.1kHz
                }
                recorder.setSampleRate(sampleRate);
            }

            // 5. 关键一步：在设置所有参数后启动记录器
            log.info("启动MP4记录器...");
            recorder.start();
            log.info("记录器成功启动");

            // 5. 创建时间戳管理器
            TimestampManager timestampManager = new TimestampManager(grabber.getFrameRate());

            // 6. 开始处理帧
            Frame frame;
            long startTime = System.currentTimeMillis();
            long recordingTimeout = 4 * 60 * 60 * 1000; // 4个小时超时
            int frameCount = 0;

            while ((frame = grabber.grab()) != null) {
                if (System.currentTimeMillis() - startTime > recordingTimeout) {
                    log.info("录制超时，自动停止");
                    result = false;
                    break;
                }
                try {
                    if (frame.image != null) {
                        frameCount++;
                        long rawTimestamp = frame.timestamp;

                        // 使用自定义时间戳管理
                        long adjustedTimestamp = timestampManager.getNextVideoTimestamp();

                        if (rawTimestamp > adjustedTimestamp) {
                            adjustedTimestamp = rawTimestamp;
                        }

                        recorder.setTimestamp(adjustedTimestamp);

                        // 记录帧
                        recorder.record(frame);

                        // 显示进度
                        if (frameCount % 100 == 0) {
                            log.info("已处理 {} 视频帧 [时间戳: {}]", frameCount, adjustedTimestamp);
                        }
                    } else if (frame.samples != null && audioChannels > 0) {
                        long audioTimestamp = timestampManager.getNextAudioTimestamp(grabber.getSampleRate());
                        recorder.setTimestamp(audioTimestamp);
                        recorder.record(frame);
                    }
                } catch (Exception e) {
                    log.error("处理帧时出错: {}", e.getMessage(), e);
                    // 特殊处理时间戳错误
                    if (e.getMessage().contains("-22")) {
                        timestampManager.reset();
                        Thread.sleep(100);
                    }
                }
            }
            log.info("录制结束! 共处理 {} 视频帧，录制结果：{}", frameCount, result);
        } catch (Exception e) {
            result= false;
            log.error("发生错误: {}", e.getMessage(), e);
            e.printStackTrace();
        } finally {
            log.info("清理资源...");
            try {
                if (recorder != null) {
                    recorder.close();
                }
            } catch (Exception e) {
                log.error("关闭记录器失败: {}", e.getMessage(), e);
            }
            try {
                if (grabber != null) {
                    grabber.close();
                }
            } catch (Exception e) {
                log.error("关闭抓取器失败: {}", e.getMessage(), e);
            }
        }
        return result;
    }

    public static void main(String[] args) {
        String host = "https://api-ivm.myhuaweicloud.com";
        String userId = "5994908220250519164352";
        String ak = "49b1d40f0870594e34e522db834233ca";
        String sk = "5f256aefd57261ae8fde307fd4547c6dcfc0cc9b091be83f0f90cb6a216bc96f";
        String token = getAccessToken(host, userId, ak, sk, null);
        System.out.println("token="+token);
//        List<DeviceDto> deviceList = getDeviceList(host, userId, token);
//        System.out.println("deviceList="+JSONObject.toJSONString(deviceList));
//        ChannelDto channelDto = getChannelByDeviceId(host, userId, token, "44000000001320000099");
        getPlaybackByDeviceId(host, userId, token, "44000000001320000202", "44000000001320000202",
                "2025-06-11 06:00:00", "2025-06-11 08:30:00");
//        System.out.println("channelDto="+JSONObject.toJSONString(channelDto));

//        List<Date> startTimeList = new ArrayList<>(Arrays.asList(
//                cn.hutool.core.date.DateUtil.parse("2025-06-10 09:35:00", "yyyy-MM-dd HH:mm:ss"),
//                cn.hutool.core.date.DateUtil.parse("2025-06-10 13:25:00", "yyyy-MM-dd HH:mm:ss")));
//        List<Date> endTimeList = new ArrayList<>(Arrays.asList(
//                cn.hutool.core.date.DateUtil.parse("2025-06-10 11:15:00", "yyyy-MM-dd HH:mm:ss"),
//                cn.hutool.core.date.DateUtil.parse("2025-06-10 15:05:00", "yyyy-MM-dd HH:mm:ss")));
//        setDeviceRecordPlan(host, userId, token, "44000000001320000099", startTimeList, endTimeList);
//        System.out.println(cn.hutool.core.date.DateUtil.format(cn.hutool.core.date.DateUtil.parse("2025-05-26 12:33:11"), "HH:mm:ss"));

//        String inputStreamUrl = "http://mgw5-iec-sjz.ivm.cn-north-4.myhuaweicloud.cn:7083/vod/vod.m3u8?token=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJhbSI6IkdCMjgxODEiLCJjIjowLCJjaGFubmVsX2lkIjoiNDQwMDAwMDAwMDEzMjAwMDAwOTkiLCJkZXZpY2VfaWQiOiI0NDAwMDAwMDAwMTMyMDAwMDA5OSIsImV0bSI6IjIwMjUtMDYtMTAgMTU6MDU6MDAiLCJleHRtYyI6MTc0OTU5NTM1NywiaSI6MSwiaXAiOiIxMjQuNzAuMTI0LjIxNyIsImxrIjoiMTc0OTUzNzc1NzU4OTA0NjI2MjhiZmE2NGIzIiwicmVxdWVzdF9pZCI6IjE5MTc5MzhkLTQ1YzYtMTFmMC05MzIxLTAyNTUwYTAwMDExYyIsInJ0IjoiYWxsIiwic3QiOjAsInN0bSI6IjIwMjUtMDYtMTAgMTM6MjU6MDAiLCJ1c2VyX2lkIjoiNTk5NDkwODIyMDI1MDUxOTE2NDM1MiJ9.xPFJw6RmwpV-yYFas_LbsZWVX9YG1blfjxKsQeeL12k&device_id=44000000001320000099&channel_id=44000000001320000099"; // 替换为你的视频流URL
//        String outputFilePath = "/Users/linchunpeng/Downloads/output.mp4";
//        convertStreamToMp4(inputStreamUrl, outputFilePath);
    }


}
