package com.bcxin.sync.task;

import cn.hutool.core.collection.CollectionUtil;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.entity.datasync.OrganizationOpenEntity;
import com.bcxin.sync.service.DataSyncToJvsService;
import com.bcxin.sync.service.DataSyncToTljService;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * description：企业开通定时任务
 * author：linchunpeng
 * date：2025/1/10
 */
@Slf4j
@Component
public class OrganizationOpenTask {

    @Autowired
    private OrganizationOpenService organizationOpenService;
    @Autowired
    private DataSyncToTljService dataSyncToTljService;
    @Autowired
    private DataSyncToJvsService dataSyncToJvsService;
    @Autowired
    private RedissonClient redissonClient;

    //每5分钟执行
    @Scheduled(cron = "0 */1 * * * ?")
    public void organizationOpenTask() {
        log.info("===================================企业开通定时任务===================================");
        //分布式锁key
        String lockKey = "REDISSON_LOCK_ORGANIZATION_OPEN_TASK";
        //取锁
        log.info("企业开通定时任务， lockKey：{}，取锁中.....", lockKey);
        RLock lock = redissonClient.getLock(lockKey);
        try {
            //加锁，并设置过期时间 300s
            if (lock.tryLock(2, 600, TimeUnit.SECONDS)) {
                log.info("取到锁");
                List<OrganizationOpenEntity> notInitList = organizationOpenService.getNotInitList(OpenType.TLJ.getCode());
                if (CollectionUtil.isNotEmpty(notInitList)) {
                    log.info("未初始化数据到陀螺匠的组织，共：{}家", notInitList.size());
                    for (OrganizationOpenEntity organizationOpenEntity : notInitList) {
                        try {
                            organizationOpenEntity.setInitStartTime(new Date());
                            String syncResult = dataSyncToTljService.sync(organizationOpenEntity.getOrganizationId());
                            organizationOpenEntity.setIsInit(1);
                            organizationOpenEntity.setInitEndTime(new Date());
                            organizationOpenEntity.setInitResult(syncResult);
                            organizationOpenService.updateById(organizationOpenEntity);
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("企业开通陀螺匠异常，企业id：{}，企业名称：{}，{}",
                                    organizationOpenEntity.getOrganizationId(), organizationOpenEntity.getOrganizationName(), e.getMessage(), e);
                        }
                    }
                } else {
                    log.info("没有未初始化数据到陀螺匠的组织");
                }
                log.info("===================================同步陀螺匠结束===================================");

                log.info("===================================同步JVS开始===================================");
                List<OrganizationOpenEntity> notInitJvsList = organizationOpenService.getNotInitList(OpenType.JVS.getCode());
                if (CollectionUtil.isNotEmpty(notInitJvsList)) {
                    log.info("未初始化数据到JVS的组织，共：{}家", notInitJvsList.size());
                    for (OrganizationOpenEntity organizationOpenEntity : notInitJvsList) {
                        try {
                            organizationOpenEntity.setInitStartTime(new Date());
                            String syncResult = dataSyncToJvsService.sync(organizationOpenEntity.getOrganizationId());
                            organizationOpenEntity.setIsInit(1);
                            organizationOpenEntity.setInitEndTime(new Date());
                            organizationOpenEntity.setInitResult(syncResult);
                            organizationOpenService.updateById(organizationOpenEntity);
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("企业开通JVS异常，企业id：{}，企业名称：{}，{}",
                                    organizationOpenEntity.getOrganizationId(), organizationOpenEntity.getOrganizationName(), e.getMessage(), e);
                        }
                    }
                } else {
                    log.info("没有未初始化数据到JVS的组织");
                }
                log.info("===================================同步JVS结束===================================");
            } else {
                log.info("没有取到锁，跳过");
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("企业开通定时任务异常，{}", e.getMessage(), e);
        } finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
            log.info("企业开通定时任务，lockKey：{}，解锁", lockKey);
        }
        log.info("===================================企业开通定时任务结束===================================");
    }

}
