package com.bcxin.sync.task;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.bcxin.sync.dtos.response.BindDeviceResponse;
import com.bcxin.sync.entity.tenant.DeviceRecordEntity;
import com.bcxin.sync.service.tenant.DeviceRecordService;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * description：下载华为云设备录像定时任务
 * author：linchunpeng
 * date：2025/6/11
 */
@Slf4j
@Component
public class DownloadHwDeviceRecordTask {

    @Autowired
    private DeviceRecordService deviceRecordService;
    @Autowired
    private RedissonClient redissonClient;

    //每2分钟执行
    @Scheduled(cron = "0 */2 * * * ?")
    public void downloadHwDeviceRecordTask() {
        log.info("===================================下载华为云设备录像定时任务===================================");
        //分布式锁key
        String lockKey = "REDISSON_LOCK_DOWNLOAD_HW_DEVICE_RECORD_TASK";
        //取锁
        log.info("下载华为云设备录像定时任务， lockKey：{}，取锁中.....", lockKey);
        RLock lock = redissonClient.getLock(lockKey);
        try {
            //加锁，并设置过期时间 300s
            if (lock.tryLock(2, 600, TimeUnit.SECONDS)) {
                log.info("取到锁");
                Date now = new Date();
                //查询昨天+今天所有的考试和设备信息
                List<BindDeviceResponse> recentlyExamDeviceList = deviceRecordService.getRecentlyExamDeviceList();
                //查询昨天+今天所有的录像记录列表
                List<DeviceRecordEntity> recentlyDeviceRecordList = deviceRecordService.getRecentlyDeviceRecordList();

                //判断需要启动下载，并且不存在下载记录的考场和设备
                if (CollectionUtil.isNotEmpty(recentlyExamDeviceList)) {
                    log.info("最近考试有：{}场", recentlyExamDeviceList.size());
                    for (BindDeviceResponse examDevice : recentlyExamDeviceList) {
                        //结束时间早于当前时间5分钟 && 设备编号和开始时间，不在录制列表里面
                        if (examDevice.getEndTime().before(DateUtil.offsetMinute(now, -5))
                                && recentlyDeviceRecordList.stream().noneMatch(record ->
                                record.getDeviceNo().equals(examDevice.getDeviceNo())
                                        && record.getStartTime().getTime() == examDevice.getStartTime().getTime())) {
                            //启动新录像记录下载
                            deviceRecordService.insertRecord(examDevice);
                        }
                    }
                }

                for (DeviceRecordEntity entity : recentlyDeviceRecordList) {
                    log.info("最近录制视频有：{}次", recentlyDeviceRecordList.size());
                    //下载失败 && 下载次数小于等于3
                    if (entity.getFileStatus() != null && entity.getDownloadCount() <= 3
                            && (entity.getFileStatus() == 0
                                || (entity.getFileStatus() == 1
                                && entity.getCreateTime().before(DateUtil.offsetHour(now, -30))))) {
                        //启动录像记录失败重试下载
                        deviceRecordService.retryDownload(entity);
                    }
                }
            } else {
                log.info("没有取到锁，跳过");
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("下载华为云设备录像定时任务异常，{}", e.getMessage(), e);
        } finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
            log.info("下载华为云设备录像定时任务，lockKey：{}，解锁", lockKey);
        }
        log.info("===================================下载华为云设备录像定时任务结束===================================");
    }

}
