package com.bcxin.sync.service.tuoluojiang;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.utils.CommonUtils;
import com.bcxin.sync.dao.mapper.tuoluojiang.EbAttendanceGroupMapper;
import com.bcxin.sync.entity.tuoluojiang.*;
import com.bcxin.sync.service.RedisCache;
import com.bcxin.sync.service.tenant.TlkAttendanceSiteBaseInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * description：陀螺匠-考勤组表
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class EbAttendanceGroupService extends ServiceImpl<EbAttendanceGroupMapper, EbAttendanceGroupEntity> {

    @Autowired
    private EbAdminService ebAdminService;
    @Autowired
    private EbAdminInfoService ebAdminInfoService;
    @Autowired
    private EbFrameAssistService ebFrameAssistService;
    @Autowired
    private EbAttendanceGroupMemberService ebAttendanceGroupMemberService;
    @Autowired
    private EbAttendanceGroupShiftService ebAttendanceGroupShiftService;
    @Autowired
    private TlkAttendanceSiteBaseInfoService tlkAttendanceSiteBaseInfoService;
    @Autowired
    private RedissonClient redissonClient;

    @Autowired
    private RedisCache redisCache;

    /**
     * description：根据驻勤点id查询考勤组
     * author：linchunpeng
     * date：2024/12/31
     */
    public Long getIdByAttendanceSiteId(String attendanceSiteId) {
        String key = "data-sync:attendance-site:group-id:" + attendanceSiteId;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString());
        }
        LambdaQueryChainWrapper<EbAttendanceGroupEntity> lqw = this.lambdaQuery();
        lqw.eq(EbAttendanceGroupEntity::getUniqued, attendanceSiteId);
        EbAttendanceGroupEntity attendanceGroupEntity = lqw.one();
        if (attendanceGroupEntity != null) {
            redisCache.setCacheObject(key, attendanceGroupEntity.getId(), 1, TimeUnit.DAYS);
            return attendanceGroupEntity.getId();
        }
        return null;
    }

    /**
     * description：创建考勤组
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public Long attendanceGroupCreate(Long entid, String attendanceSiteId, String itemAttendanceSiteName,
                                      String itemAttendanceSiteFullAddress, BigDecimal itemAttendanceSiteLongitude,
                                      BigDecimal itemAttendanceSiteLatitude, BigDecimal itemPerformRange,
                                      String itemPrincipal, String itemAttType) {
        //分布式锁key
        String lockKey = "REDISSON_LOCK_ATTENDANCE_GROUP_CREATE_" + attendanceSiteId;
        //取锁
        log.info("创建考勤组， lockKey：{}，取锁中.....", lockKey);
        RLock lock = redissonClient.getLock(lockKey);
        //加锁，并设置过期时间 30s
        lock.lock(30, TimeUnit.SECONDS);
        log.info("取到锁");
        try {
            List<EbAdminEntity> adminEntityList = new ArrayList<>();
            if (StringUtils.isNotBlank(itemPrincipal)) {
                List<String> employeeIdList = Arrays.stream(itemPrincipal.split(";")).collect(Collectors.toList());
                adminEntityList = ebAdminService.getListByEmployeeIdList(employeeIdList, entid);
            } else {
                EbAdminEntity entAdmin = ebAdminService.getEntAdmin(entid);
                if (entAdmin != null) {
                    adminEntityList.add(entAdmin);
                }
            }
            if (CollectionUtil.isNotEmpty(adminEntityList)) {
                boolean isNew = false;
                EbAttendanceGroupEntity entity;
                LambdaQueryChainWrapper<EbAttendanceGroupEntity> lqw = this.lambdaQuery();
                lqw.eq(EbAttendanceGroupEntity::getUniqued, attendanceSiteId);
                List<EbAttendanceGroupEntity> list = lqw.list();
                if (CollectionUtil.isNotEmpty(list)) {
                    //已存在则更新
                    entity = list.get(0);
                } else {
                    //不存在则新增
                    isNew = true;
                    entity = new EbAttendanceGroupEntity();
                    entity.setEntid(entid);
                    entity.setUniqued(attendanceSiteId);
                }
                entity.setEntid(entid);
                entity.setName(itemAttendanceSiteName);
                entity.setAddress(itemAttendanceSiteFullAddress != null ? itemAttendanceSiteFullAddress : "");
                entity.setLat(itemAttendanceSiteLatitude != null ? itemAttendanceSiteLatitude.toString() : "");
                entity.setLng(itemAttendanceSiteLongitude != null ? itemAttendanceSiteLongitude.toString() : "");
                entity.setEffectiveRange(itemPerformRange != null ? itemPerformRange.intValue() : 0);
                entity.setLocationName(itemAttendanceSiteName);
                entity.setRepairAllowed(0);
                entity.setRepairType("");
                entity.setUid(adminEntityList.get(0).getId());

                this.saveOrUpdate(entity);

                //考勤组管理员
                if (isNew) {
                    //新增
                    for (EbAdminEntity admin : adminEntityList) {
                        ebAttendanceGroupMemberService.addMember(entid, entity.getId(), admin.getId(), 2, true);
                    }
                } else {
                    //修改
                    List<EbAttendanceGroupMemberEntity> memberEntityList = ebAttendanceGroupMemberService.getManagerByGroupId(entity.getId());
                    if (CollectionUtil.isNotEmpty(memberEntityList)) {
                        List<Long> adminIdList = adminEntityList.stream().map(EbAdminEntity::getId).collect(Collectors.toList());
                        for (EbAttendanceGroupMemberEntity memberEntity : memberEntityList) {
                            if (!adminIdList.contains(memberEntity.getMember())) {
                                ebAttendanceGroupMemberService.deleteMember(memberEntity.getEntid(), memberEntity.getGroupId(), memberEntity.getMember(), memberEntity.getType());
                            }
                        }
                    }
                    for (EbAdminEntity admin : adminEntityList) {
                        ebAttendanceGroupMemberService.addMember(entid, entity.getId(), admin.getId(), 2, true);
                    }
                }
                //缓存驻勤点类型
                String key = "data-sync:attendance-site:att-type:" + attendanceSiteId;
                redisCache.setCacheObject(key, itemAttType, 7, TimeUnit.DAYS);

                return entity.getId();
            }
            log.info("驻勤点负责人不在陀螺匠中，请排查数据");
            return null;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("创建考勤组异常，{}", e.getMessage(), e);
        } finally {
            if (lock.isLocked()) {
                lock.unlock();
            }
            log.info("创建考勤组，lockKey：{}，解锁", lockKey);
        }
        return null;
    }

    /**
     * description：删除考勤组
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public void attendanceGroupDelete(List<String> itemAttendanceSiteIdList) {
        LambdaQueryChainWrapper<EbAttendanceGroupEntity> lqw = this.lambdaQuery();
        lqw.in(EbAttendanceGroupEntity::getUniqued, itemAttendanceSiteIdList);
        List<EbAttendanceGroupEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            Map<Long, String> groupIdAttenSiteIdMap = new HashMap<>();
            for (EbAttendanceGroupEntity ebAttendanceGroupEntity : list) {
                //删除考勤组
                ebAttendanceGroupEntity.setDeletedAt(new Date());
                groupIdAttenSiteIdMap.put(ebAttendanceGroupEntity.getId(), ebAttendanceGroupEntity.getUniqued());
            }
            List<Long> groupIdList = list.stream().map(EbAttendanceGroupEntity::getId).collect(Collectors.toList());
            //删除考勤组人员
            List<Long> excludeAdminIdList = new ArrayList<>();
            List<EbAttendanceGroupMemberEntity> memberEntityList = ebAttendanceGroupMemberService.getByGroupIdList(groupIdList);
            if (CollectionUtil.isNotEmpty(memberEntityList)) {
                for (EbAttendanceGroupMemberEntity memberEntity : memberEntityList) {
                    memberEntity.setDeletedAt(new Date());
                    //判断是否临保考勤组，如果是，删除在其他组的排除考勤成员
                    String itemAttType = tlkAttendanceSiteBaseInfoService.getItemAttType(groupIdAttenSiteIdMap.get(memberEntity.getGroupId()));
                    if (CommonUtils.isLinBao(itemAttType)) {
                        excludeAdminIdList.add(memberEntity.getMember());
                    }
                }
                ebAttendanceGroupMemberService.updateBatchById(memberEntityList);
                ebAttendanceGroupMemberService.deleteMemberExclude(excludeAdminIdList);
            }
            //删除考勤组班次
            List<EbAttendanceGroupShiftEntity> shiftEntityList = ebAttendanceGroupShiftService.getByGroupIdList(groupIdList);
            if (CollectionUtil.isNotEmpty(shiftEntityList)) {
                for (EbAttendanceGroupShiftEntity shiftEntity : shiftEntityList) {
                    shiftEntity.setDeletedAt(new Date());
                }
                ebAttendanceGroupShiftService.updateBatchById(shiftEntityList);
            }
            this.updateBatchById(list);

            //如果是临保 && 包含外部企业人员，需要离职
            if (CollectionUtil.isNotEmpty(excludeAdminIdList)) {
                List<EbAdminEntity> ebAdminEntityList = ebAdminService.listByIds(excludeAdminIdList);
                if (CollectionUtil.isNotEmpty(ebAdminEntityList)) {
                    for (EbAdminEntity adminEntity : ebAdminEntityList) {
                        EbAdminInfoEntity adminInfoEntity = ebAdminInfoService.getByUserIdAndEntid(adminEntity.getUid(), adminEntity.getEntid());
                        if (adminInfoEntity != null && adminInfoEntity.getIsPart() == 1 && adminInfoEntity.getType() != 4) {
                            //如果是在职需要改为离职
                            adminInfoEntity.setType(4);
                            adminInfoEntity.setQuitTime(DateUtil.formatDateTime(new Date()));
                            ebAdminInfoService.updateById(adminInfoEntity);

                            //删除部门人员关联
                            ebFrameAssistService.frameAssistDelete(adminEntity.getEntid(), adminEntity.getId());
                        }
                    }
                }
            }
        }
    }

}