package com.bcxin.runtime.domain.syncs.dtos;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.annotation.JSONField;
import com.bcxin.runtime.domain.constants.FieldNames;
import com.bcxin.runtime.domain.syncs.enums.ProcessedStatus;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URLDecoder;
import java.util.*;
import java.util.stream.Collectors;

@Data
public class DataSetDto {
    private String tableName;
    private String sql;
    private Date fetchedModifiedTime;
    private Date exportedTime;
    private int totalCount;
    private Collection<Column> columns;
    private Collection<Row> rows;

    private Collection<String> deletedIds;

    public DataSetDto() {
    }

    public static DataSetDto create(
            String tableName, int totalCount, Collection<Column> columns,
            Collection<Row> rows) {
        DataSetDto record = new DataSetDto();
        record.setTableName(tableName);
        record.setTotalCount(totalCount);
        record.setColumns(columns);
        record.setRows(rows);

        return record;
    }

    public void markInfo(String sql, Date fetchedModifiedTime, Date exportedTime) {
        this.setSql(sql);
        this.setExportedTime(exportedTime);
        this.setFetchedModifiedTime(fetchedModifiedTime);
    }

    @Data
    public static class Column {
        private boolean isFile;
        private String name;

        public static Column create(String name) {
            Column column = new Column();
            //String attachValue = itemJSONObject.getString("AttachmentUploadField");
            //String imageUploadValue = itemJSONObject.getString("imageUploadValue");

            if ("AttachmentUploadField".equalsIgnoreCase(name) || "imageUploadValue".equals(name)) {
                column.setFile(true);
            }

            column.setName(name);

            return column;
        }

        public boolean isId() {
            return "id".equalsIgnoreCase(this.getName());
        }

        public boolean isDomainId() {
            return "DOMAINID".equalsIgnoreCase(this.getName().replace("_", ""));
        }
    }

    @Data
    public static class Row {
        private final static String DOMAIN_ID = "DOMAIN_ID";
        private Object id;
        private Map<String, Object> parameters;
        private Collection<FieldValue> fieldValues;
        private ProcessedStatus status;
        private Date processedTime;
        private boolean convertPK = false;

        public static Row create(Object id,
                                 Collection<FieldValue> fieldValues) {
            Row rw = new Row();
            rw.setId(id);
            rw.setFieldValues(fieldValues);

            return rw;
        }

        public Object getParameterValue(String key) {
            if (this.getParameters() == null) {
                return null;
            }

            if (!this.getParameters().containsKey(key)) {
                return null;
            }

            return this.getParameters().get(key);
        }

        public String getDomainId() {
            Object value = this.getParameterValue(DOMAIN_ID);
            if (value == null) {
                return null;
            }

            return String.valueOf(value);
        }

        public void setDomainId(String value) {
            this.addParameter(DOMAIN_ID, value);
        }

        public void addParameter(String key, Object value) {
            Map<String, Object> pm = this.getParameters();
            if (pm == null) {
                pm = new HashMap<>();
                this.setParameters(pm);
            }

            pm.put(key, value);
        }

        public void changeStatus(ProcessedStatus status) {
            this.setStatus(status);
            this.setProcessedTime(new Date());
        }
    }


    @Data
    public static class FieldValue {
        private static final Logger logger = LoggerFactory.getLogger(FieldValue.class);
        private String name;
        private Object value;
        private String originalValue;
        private Collection<FieldNames.PathOption> pathOptions;

        @JSONField(serialize = false)
        public boolean getIsFileValue() {
            if (FieldNames.isFile(getName())) {
                return true;
            }

            return false;
        }

        @JSONField(serialize = false)
        public boolean isValidFile() {
            if (this.getIsFileValue()) {
                boolean flag = this.getValue() != null && !"".equalsIgnoreCase(String.valueOf(this.getValue())) &&
                        !"null".equalsIgnoreCase(String.valueOf(this.getValue()));
                if (!flag) {
                    return false;
                }

                try {
                    return String.valueOf(this.getValue()).contains("[") &&
                            String.valueOf(this.getValue()).contains("]") &&
                            String.valueOf(this.getValue()).contains("{") &&
                            String.valueOf(this.getValue()).contains("}");
                } catch (Exception ex) {
                    System.err.println(String.format("字段信息:%s", this.getName()));
                    ex.printStackTrace();
                }
            }

            return false;
        }

        public static FieldValue create(String sitePrefix, String domainId, String rowId, String name, Object value) {
            FieldValue fieldValue = new FieldValue();
            fieldValue.setName(name);
            Object formattedValue = value;
            String originalValue = String.valueOf(value);

            if (fieldValue.getIsFileValue()) {
                if (!StringUtils.isEmpty(originalValue) && !"null".equalsIgnoreCase(originalValue)) {
                    // 针对文件字段,
                    try {
                        originalValue = URLDecoder.decode(originalValue, "UTF-8");
                    } catch (Exception e) {
                        logger.error("同步表文件相关字段转码异常，文件名为：{}", name, e);
                    }
                    formattedValue = FieldNames.formattedFileValue(sitePrefix, domainId, rowId, originalValue);
                    if (formattedValue != null && formattedValue.toString().contains("[")) {
                        try {
                            List<FieldNames.PathOption> pathOptions =
                                    JSON.parseArray((String) formattedValue, FieldNames.PathOption.class);
                            fieldValue.setPathOptions(pathOptions);
                        } catch (Exception ex) {
                            logger.error(String.format("初始化字段: %s, 尝试获取文件值信息报错", name), ex);
                            ex.printStackTrace();
                        }
                    }
                }
            }

            fieldValue.setOriginalValue(originalValue);
            fieldValue.setValue(formattedValue);

            return fieldValue;
        }

        @JSONField(serialize = false)
        public Collection<String> getFileUrls() {
            if (!this.isValidFile() || this.getPathOptions() == null) {
                return null;
            }

            return this.getPathOptions().stream().map(ix -> ix.getOriginalPath()).collect(Collectors.toList());
        }

        @JSONField(serialize = false)
        public Collection<String> getFileIds() {
            if (!this.isValidFile() || this.getPathOptions() == null) {
                return null;
            }

            return this.getPathOptions().stream().map(ix -> ix.getUid()).collect(Collectors.toList());
        }
    }
}
