package com.bcxin.runtime.domain.repositories.components;

import com.bcxin.runtime.domain.repositories.permissions.repositories.JpaDataPermissionScopeRepository;
import com.bcxin.saas.core.components.*;
import com.bcxin.saas.core.components.dtos.DataPermissionScopeDTO;
import com.bcxin.runtime.domain.permissions.entities.DataPermissionScopeEntity;
import com.bcxin.saas.core.components.dtos.DataPermissionType;
import com.bcxin.saas.core.enums.EnvironmentType;
import com.bcxin.saas.core.exceptions.SaasNoSupportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.Collections;

public class DataPermissionScopeProviderImpl implements DataPermissionScopeProvider {
    private static final Logger logger = LoggerFactory.getLogger(DataPermissionScopeProviderImpl.class);
    private final JsonProvider jsonProvider;
    private final ThreadContextManager threadContextManager;
    private final CookieProvider cookieProvider;
    private final DistributedCacheProvider distributedCacheProvider;
    private final JpaDataPermissionScopeRepository dataPermissionScopeRepository;
    private final DataPermissionScopeConfig dataPermissionScopeConfig;

    public DataPermissionScopeProviderImpl(JsonProvider jsonProvider,
                                           ThreadContextManager threadContextManager,
                                           CookieProvider cookieProvider,
                                           DistributedCacheProvider distributedCacheProvider,
                                           JpaDataPermissionScopeRepository dataPermissionScopeRepository,
                                           DataPermissionScopeConfig dataPermissionScopeConfig) {
        this.threadContextManager = threadContextManager;
        this.cookieProvider = cookieProvider;
        this.distributedCacheProvider = distributedCacheProvider;
        this.jsonProvider = jsonProvider;
        this.dataPermissionScopeRepository = dataPermissionScopeRepository;
        this.dataPermissionScopeConfig = dataPermissionScopeConfig;
    }

    /**
     * 默认情况下, 无需走redis来获取数据信息, 因为cookie中的值会设置为false; 因此, 杜绝大部分企业还需要走redis的情况
     *
     * @param id
     * @param permissionType
     * @return
     */
    @Override
    public DataPermissionScopeDTO getById(String appId,String employeeId, String id, DataPermissionType permissionType) {
        return getByAppIdAndId(appId,employeeId,id, permissionType);
    }

    @Override
    public Collection<String> getScopesById(String appId,String employeeId, String id, DataPermissionType permissionType) {
        return getScopesByAppIdAndId(appId, employeeId,id, permissionType);
    }

    @Override
    public DataPermissionScopeDTO getByAppIdAndId(String appId,
                                                  String employeeId,
                                                  String id,
                                                  DataPermissionType permissionType) {
        /**
         * 仅行业信息需要做此限制
         */
        if (StringUtils.hasLength(appId) &&
                !"__DOoeSJp26wVqbyYYf50".equalsIgnoreCase(appId) &&
                !"__IHG81vhqOehuUwpCLin".equalsIgnoreCase(appId) &&
                !"__JvOHkqjhV41mUo4ITRv".equalsIgnoreCase(appId)&&
                !"__N4pdLMmSZIrmOGBmb5i".equalsIgnoreCase(appId) &&
                !"__E90fgpplhOsklLI9mr4".equalsIgnoreCase(appId)
        ) {
            return DataPermissionScopeDTO.empty(id);
        }

        if (dataPermissionScopeConfig.getEnvironmentType() == EnvironmentType.Government_Inner &&
                dataPermissionScopeConfig.isAllowedAllData()) {
            if (!dataPermissionScopeConfig.isConditionLimitedApp(appId)) {
                return DataPermissionScopeDTO.createForAllPermissions(id);
            }
        }


        String cacheKey = null;
        String secondLevelCacheKey = null;
        DataPermissionScopeDTO data = null;
        String scopedId = null;
        if(permissionType==DataPermissionType.Enterprise) {
            scopedId = id;
            cacheKey = DataPermissionScopeConfig.getPermissionCacheKey(permissionType, id);
        }else {
            scopedId = employeeId;
            secondLevelCacheKey = DataPermissionScopeConfig.getPermissionScopeCacheKey(permissionType, employeeId);
        }

        String cookieKey = String.format("data.scopes.permission.%s.%s.flag", permissionType, scopedId);
        String cookieValue = this.cookieProvider.get(cookieKey);
        if ("false".equalsIgnoreCase(cookieValue)) {
            /**
             * 无权限的用户; 直接返回企业Id
             */
            return DataPermissionScopeDTO.empty(id);
        }

        if(permissionType==DataPermissionType.Enterprise) {
            data = this.threadContextManager.get(cacheKey);
        }else {
            cacheKey = distributedCacheProvider.get(secondLevelCacheKey);
            if (StringUtils.hasLength(cacheKey)) {
                data = this.threadContextManager.get(cacheKey);

                if (data == null) {
                    data = this.distributedCacheProvider.get(cacheKey);
                }
            }
        }

        if (data == null) {
             {
                DataPermissionScopeDTO dataPermissionScope = null;

                /**
                 * 针对集团公司才需要配置每个组织Id作为PermissionDataScope的一条数据;
                 */
                if(permissionType==DataPermissionType.Enterprise) {
                    dataPermissionScope = this.distributedCacheProvider.get(cacheKey);
                }

                /**
                 * 非集团这个为空
                 */
                if (dataPermissionScope == null) {
                    dataPermissionScope = getDataPermissionScopeDTO(id,employeeId, permissionType);

                    if (dataPermissionScope == null) {
                        dataPermissionScope = DataPermissionScopeDTO.empty(id);
                    }

                    if (permissionType == DataPermissionType.Enterprise) {
                        this.distributedCacheProvider.put(cacheKey, dataPermissionScope, 10 * 60 * 60);
                    } else {
                        /**
                         * 非empty的数据
                         */
                        if (StringUtils.hasLength(dataPermissionScope.getSearchIndexCode())) {
                            cacheKey = DataPermissionScopeConfig.getPermissionCacheKey(permissionType, dataPermissionScope.getSearchIndexCode());
                            /**
                             * 设置缓存数据
                             */
                            distributedCacheProvider.put(secondLevelCacheKey, cacheKey);
                            this.distributedCacheProvider.put(cacheKey, dataPermissionScope, 10 * 60 * 60);
                        }
                    }
                }

                if (!StringUtils.hasLength(cookieValue) || !dataPermissionScope.isCrossPermission()) {
                    this.cookieProvider.add(cookieKey, String.valueOf(dataPermissionScope.isCrossPermission()));
                }

                data = dataPermissionScope;
            }
        }

        return data;
    }

    @Override
    public Collection<String> getScopesByAppIdAndId(String appId, String employeeId,String id, DataPermissionType permissionType) {
        DataPermissionScopeDTO dataPermissionScope = getById(appId, employeeId,id, permissionType);
        if (dataPermissionScope == null) {
            return Collections.EMPTY_LIST;
        }

        return dataPermissionScope.getScopes();
    }


    private DataPermissionScopeDTO getDataPermissionScopeDTO(String orgId,String employeeId, DataPermissionType permissionType) {
        DataPermissionScopeEntity scopeEntity = null;
        switch (permissionType) {
            case Enterprise: {
                scopeEntity =
                        this.dataPermissionScopeRepository
                                .findById(orgId).orElse(null);
                break;
            }
            case Securtity:
            case Department: {
                if (permissionType == DataPermissionType.Department) {
                    scopeEntity = this.dataPermissionScopeRepository.getPermissionScopesByEmployeeId(employeeId);
                } else {
                    scopeEntity =
                            this.dataPermissionScopeRepository.getPermissionScopesBySecurityOrgId(orgId);
                }
                break;
            }
            default: {
                throw new SaasNoSupportException();
            }
        }

        if (scopeEntity == null) {
            return DataPermissionScopeDTO.empty(orgId);
        }

        Collection<String> scopes =
                this.jsonProvider.getDataList(scopeEntity.getContent(), String.class);

        return DataPermissionScopeDTO.create(
                scopeEntity.getId(),
                scopeEntity.getSearchIndexCode(),
                scopes,
                Collections.singleton(orgId)
        );
    }
}
