package com.bcxin.common.apis.configs;

import cn.myapps.designtime.requests.RefreshDesignRequest;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.components.CacheConst;
import com.bcxin.components.TenantContext;
import com.bcxin.saas.core.components.DataPermissionScopeConfig;
import com.bcxin.saas.core.components.DistributedCacheProvider;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.components.dtos.DataPermissionType;
import com.bcxin.saas.core.enums.HttpMethod;
import com.bcxin.saas.core.utils.CompleteFurtherUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@Configuration
/**
 * 仅内网和互联网可用（其他地方采用）
 */
@ConditionalOnProperty(value = "spring.kafka.bootstrap-servers",matchIfMissing = false )
public class KafkaConsumerListener {
    private static final Logger logger = LoggerFactory.getLogger(KafkaConsumerListener.class);
    private final JsonProvider jsonProvider;
    private final DistributedCacheProvider distributedCacheProvider;
    private final NamedParameterJdbcTemplate jdbcTemplate;
    private final DiscoveryClient discoveryClient;

    private final String TOPIC_OBPM2_T_USER_DEPARTMENT_ROLE_SET = "obpm2.binlog-cdc.topic.v2.t_user_department_role_set";
    private final String TOPIC_OBPM2_TENANT_ORGANIZATION_RELATIONSHIPS  = "obpm2.binlog-cdc.topic.v2.tenant_organization_relationships";

    public static final String TOPIC_OBPM2_TENANT_ORGANIZATIONS = "obpm2.binlog-cdc.topic.v2.tenant_organizations";

    private final String TOPIC_OBPM2_TENANT_EMPLOYEES = "obpm2.binlog-cdc.topic.v2.tenant_employees";

    public static final String TOPIC_OBPM2_TENANT_EXTERNAL_MEMBERS = "obpm2.binlog-cdc.topic.v2.external_members";

    private final String REFRESH_ORG_DATA_SCOPE_PERMISSION_DELETE_SQL=
            "delete from tenant_data_permissions where id in (:organizationIds)";

    private final String REFRESH_ORG_DATA_SCOPE_PERMISSION_UPDATE_SQL=
            "insert into tenant_data_permissions\n" +
                    "(id,content,created_time,last_updated_time,supervise_region_code,industry_code,institutional_code)\n" +
                    "SELECT \n" +
                    "  pt.organization_id,pt.content,CURRENT_TIMESTAMP,CURRENT_TIMESTAMP,pt.supervise_region_code,pt.industry_code,pt.institutional_code\n" +
                    "FROM (\n" +
                    "  select \n" +
                    "\t    CONCAT('[',GROUP_CONCAT( DISTINCT CONCAT('\\\"',rp.selected_organization_id,'\\\"')),']') as content,rp.organization_id,o.supervise_region_code,o.industry_code,o.institutional_code\n" +
                    "\tfrom\n" +
                    "   tenant_organizations o left join tenant_organization_relationships rp on o.id=rp.organization_id\n" +
                    "\t where o.id=:organizationIds and o.`level` & 2>0 and rp.is_deleted = 0  and rp.`status` = 0 \n" +
                    "\tgroup by rp.organization_id\n" +
                    ") AS pt\n" +
                    "ON DUPLICATE KEY UPDATE content=pt.content,last_updated_time=CURRENT_TIMESTAMP";

    public KafkaConsumerListener(JsonProvider jsonProvider, DistributedCacheProvider distributedCacheProvider,
                                 NamedParameterJdbcTemplate jdbcTemplate, DiscoveryClient discoveryClient) {
        this.jsonProvider = jsonProvider;
        this.distributedCacheProvider = distributedCacheProvider;
        this.jdbcTemplate = jdbcTemplate;
        this.discoveryClient = discoveryClient;

        logger.error("调试: 初始化KafkaConsumerListener 信息");
    }

    @DependsOn("batchFactory")
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-user-department-role-set",
            topics = {
                    TOPIC_OBPM2_T_USER_DEPARTMENT_ROLE_SET
            }, groupId = "${spring.kafka.consumer.group-id}-relative-changes")
    public void ackUserDepartmentRoleSetListener(
            List<ConsumerRecord<String, String>> records, Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }


        boolean allowed2CommitAtFinial = true;
        try {
            Collection<String> employeeIds =
                    records.stream()
                            .flatMap(ii -> extractValue(ii.value(), "USERID").stream())
                            .filter(ii -> StringUtils.hasLength(ii))
                            .collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(employeeIds)) {
                Collection<String> cacheKeys =
                        employeeIds.stream().map(ii -> CacheConst.getUserRoleDeptKmsRelationCacheKey(ii))
                        .collect(Collectors.toList());
                this.distributedCacheProvider.remove(cacheKeys);
            }

            
            logger.error("成功从缓存中清除用户({})的授权信息", employeeIds.stream().collect(Collectors.joining(",")));
        } catch (Exception ex) {
            logger.error("消费用户角色信息发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @DependsOn("batchFactory")
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-external_members",
            topics = {
                    TOPIC_OBPM2_TENANT_EXTERNAL_MEMBERS
            }, groupId = "${spring.kafka.consumer.group-id}-external_members")
    public void ackExternalMemberListener(
            List<ConsumerRecord<String, String>> records, Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<Map<String, Object>> parameters =
                    records.stream().map(ii -> {
                        JSONObject data = this.jsonProvider.getData(ii.value(), JSONObject.class);
                        JSONObject after = data.getJSONObject("after");
                        JSONObject before = data.getJSONObject("before");
                        JSONObject dataNode = after == null ? before : after;

                        Map<String, Object> mpValue = new HashMap<>();
                        mpValue.put("id", dataNode.get("id"));
                        return mpValue;
                    }).filter(ii -> ii != null).collect(Collectors.toList());

            if(!CollectionUtils.isEmpty(parameters)) {
                String insertOrUpdateSql =
                        "INSERT INTO `t_user`(`ID`, `name`, `loginno`, `LOGINPWD`, `domainid`," +
                                "`LEVELS`,`STATUS`,`LIAISON_OFFICER`,`TELEPHONEPUBLIC`,`TELEPHONEPUBLIC2`,`EMAILPUBLIC`,`USERINFOPUBLIC`,`DEFAULTDEPARTMENT`," +
                                "`PERMISSION_TYPE`,`field11`,`field12`,`ISDOMAINUSER`,`resource_type`)" +
                                " select m.id,u.`name`,m.id,m.id,m.reference_number,0,1,1,1,1,1,1," +
                                " (select t.id from t_department t where t.DOMAIN_ID=m.reference_number limit 1),'public','','',0,1\n" +
                                " from external_members m join tenant_users u on m.tenant_user_id=u.id " +
                                " where m.id=:id ON DUPLICATE KEY UPDATE id=m.id,name=u.`name`";

                Map<String, Object>[] mapArray = parameters.toArray(new Map[0]);
                this.jdbcTemplate.batchUpdate(insertOrUpdateSql, mapArray);
            }
        } catch (Exception ex) {
            logger.error("新增t_user人员信息异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @DependsOn("batchFactory")
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-tenant_employees",
            topics = {
                    TOPIC_OBPM2_TENANT_EMPLOYEES
            }, groupId = "${spring.kafka.consumer.group-id}-tenant_employees-changes")
    public void ackTenantEmployeesListener(
            List<ConsumerRecord<String, String>> records, Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }


        boolean allowed2CommitAtFinial = true;
        try {
            /**
             * 查找存在组织管理员身份发生变更的信息
             */
            Collection<String> employeeIds =
                    records.stream()
                            .filter(ii->{
                                Collection<String> is_domain_admins = extractValue(ii.value(),"is_domain_admin");
                                if(CollectionUtils.isEmpty(is_domain_admins)) {
                                    return false;
                                }

                                return is_domain_admins.stream().distinct().count()>1;
                            })
                            .flatMap(ii -> extractValue(ii.value(), "id").stream())
                            .filter(ii -> StringUtils.hasLength(ii))
                            .collect(Collectors.toSet());

            if (!CollectionUtils.isEmpty(employeeIds)) {
                List<ServiceInstance> runtimeInstances =
                        this.discoveryClient.getInstances("obpm-runtime");
                CompleteFurtherUtils.execute(runtimeInstances, (instance) -> {
                    postRefreshInstanceUserCache(HttpMethod.POST, instance, "%s/obpm/api/caches/user/refresh", employeeIds);
                });
            }

            logger.error("成功从缓存中清除用户([组织管理员与非组织管理员之间切换的用户])({})的授权信息", employeeIds.stream().collect(Collectors.joining(",")));
        } catch (Exception ex) {
            logger.error("消费用户角色信息发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @DependsOn("batchFactory")
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-organization-relation-ships",
            topics = {
                    TOPIC_OBPM2_TENANT_ORGANIZATION_RELATIONSHIPS
            }, groupId = "${spring.kafka.consumer.group-id}-relative-changes")
    public void ackTenantOrganizationRelationShipsListener(
            List<ConsumerRecord<String, String>> records, Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }


        boolean allowed2CommitAtFinial = true;
        try {
            Collection<String> organizationIds =
                    records.stream()
                            .flatMap(ii -> extractValue(ii.value(), "organization_id").stream())
                            .filter(ii -> StringUtils.hasLength(ii))
                            .collect(Collectors.toSet());
            int affectedRows = 0;
            if (!CollectionUtils.isEmpty(organizationIds)) {
                Map<String, Collection<String>> params = new HashMap<>();
                params.put("organizationIds", organizationIds);

                this.jdbcTemplate.update(REFRESH_ORG_DATA_SCOPE_PERMISSION_DELETE_SQL, params);
                affectedRows = this.jdbcTemplate.update(REFRESH_ORG_DATA_SCOPE_PERMISSION_UPDATE_SQL, params);

                Collection<String> cacheKeys =
                        organizationIds.stream().map(ii -> DataPermissionScopeConfig.getPermissionCacheKey(DataPermissionType.Enterprise, ii))
                                .collect(Collectors.toSet());
                this.distributedCacheProvider.remove(cacheKeys);
            }

            logger.error("成功归集了集团({})的数据权限相关信息; 影响的行数={}", organizationIds.stream().collect(Collectors.joining(",")), affectedRows);
        } catch (Exception ex) {
            logger.error("消费集团数据权限信息发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }


    @DependsOn("batchFactory")
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-organizations",
            topics = {
                    TOPIC_OBPM2_TENANT_ORGANIZATIONS
            }, groupId = "${spring.kafka.consumer.group-id}-organizations")
    public void ackTenantOrganizationsListener(
            List<ConsumerRecord<String, String>> records, Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }


        boolean allowed2CommitAtFinial = true;
        try {
            Collection<String> cacheKeys =
                    records.stream()
                            .flatMap(ii -> extractValue(ii.value(), "id").stream())
                            .filter(ii -> StringUtils.hasLength(ii))
                            .map(ii -> TenantContext.getOrgContextCacheKey(ii))
                            .collect(Collectors.toSet());
            int affectedRows = 0;
            if (!CollectionUtils.isEmpty(cacheKeys)) {
                this.distributedCacheProvider.remove(cacheKeys);
            }

            logger.error("成功消费组织变更的缓存; 影响的行数={}", cacheKeys.stream().collect(Collectors.joining(",")), affectedRows);
        } catch (Exception ex) {
            logger.error("消费组织变更的缓存", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    private Collection<String> extractValue(String debeziumJsonValue, String field) {
        if (!StringUtils.hasLength(debeziumJsonValue)) {
            return Collections.EMPTY_LIST;
        }

        JSONObject rootNode =
                this.jsonProvider.getData(debeziumJsonValue, JSONObject.class);

        if (rootNode == null) {
            return Collections.EMPTY_LIST;
        }

        Collection<JSONObject> allNodes = new HashSet<>();
        JSONObject before = rootNode.getJSONObject("before");
        if (before != null) {
            allNodes.add(before);
        }
        JSONObject after = rootNode.getJSONObject("after");
        if (after != null) {
            allNodes.add(after);
        }

        Collection<String> employeeIds =
                allNodes.stream().map(ii -> {
                            String selectedKey =
                                    ii.keySet().stream().filter(fi -> fi.equalsIgnoreCase(field)).findFirst().orElse(null);
                            if (StringUtils.hasLength(selectedKey)) {
                                return ii.getString(selectedKey);
                            }

                            return null;
                        }).filter(ii -> StringUtils.hasLength(ii))
                        .collect(Collectors.toSet());

        return employeeIds;
    }

    private void postRefreshInstanceUserCache(
            HttpMethod method,
            ServiceInstance instance, String urlFormat,Object params ) {
        String targetUrl = String.format(urlFormat,instance.getUri());
        try {
            CloseableHttpClient httpClient = null;
            try {
                httpClient = HttpClientBuilder.create().build();

                HttpEntityEnclosingRequestBase put = new HttpPost(targetUrl);
                if(method==HttpMethod.PUT) {
                    put = new HttpPut(targetUrl);
                }

                String json = this.jsonProvider.getJson(params);
                put.setEntity(new StringEntity(json, ContentType.APPLICATION_JSON));
                CloseableHttpResponse response = httpClient.execute(put);
                String result = EntityUtils.toString(response.getEntity(), "UTF-8");

                logger.error("推送表单数据至{}-数据={} 保存结果为:{}-{}",
                        put.getURI(), json, response.getStatusLine(), result);
            } finally {
                if (httpClient != null) {
                    httpClient.close();
                }
            }
        } catch (Exception ex) {
            logger.error("刷新runtime({})组件缓存发生异常:{}", targetUrl, this.jsonProvider.getJson(params), ex);
        }
    }
}
