package com.bcxin.ars.util.ftp;

import java.io.*;
import java.net.SocketException;
import java.nio.file.Paths;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.*;

/****
 * FTP连接工具类
 * @author  subh
 */
public class FtpUtils {
    /***
     * 日志
     */
    private final static Log logger = LogFactory.getLog(FtpUtils.class);
    /**
     * 获取FTPClient对象
     * @param ftpHost
     *            FTP主机服务器
     * @param ftpPassword
     *            FTP 登录密码
     * @param ftpUserName
     *            FTP登录用户名
     * @param ftpPort
     *            FTP端口 默认为21
     * @return
     */
    public static FTPClient getFTPClient(String ftpHost, String ftpUserName,
                                         String ftpPassword, int ftpPort) {
        FTPClient ftpClient = new FTPClient();
        try {
            ftpClient = new FTPClient();
            ftpClient.connect(ftpHost, ftpPort);// 连接FTP服务器
            ftpClient.login(ftpUserName, ftpPassword);// 登陆FTP服务器
            if (!FTPReply.isPositiveCompletion(ftpClient.getReplyCode())) {
                logger.info("未连接到FTP，用户名或密码错误。");
                ftpClient.disconnect();
            } else {
                logger.info("FTP连接成功。");
            }
        } catch (SocketException e) {
            logger.error(e.getMessage(),e);
        } catch (IOException e) {
            logger.error(e.getMessage(),e);
        }
        return ftpClient;
    }

    /**
     * 下载文件
     * @param ftpHost ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort ftp服务员器端口号
     * @param ftpPath  ftp文件存放物理路径
     * @param fileName 文件路径
     */
    public static InputStream downloadFile(String ftpHost, String ftpUserName,
                                           String ftpPassword, int ftpPort, String ftpPath,
                                           String fileName) {
        FTPClient ftpClient = null;
        InputStream is = null;
        try {
            ftpClient = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            ftpClient.setControlEncoding("UTF-8"); // 中文支持
            ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
            ftpClient.enterLocalPassiveMode();
            ftpClient.changeWorkingDirectory(ftpPath);
            is  = ftpClient.retrieveFileStream(fileName);
        } catch (FileNotFoundException e) {
            logger.error("没有找到" + ftpPath + "文件");
            logger.error(e.getMessage(),e);
        } catch (SocketException e) {
            logger.error("连接FTP失败.");
            logger.error(e.getMessage(),e);
        } catch (IOException e) {
            logger.error("文件读取错误。");
            logger.error(e.getMessage(),e);
        }finally {
            try {
                if (ftpClient != null) ftpClient.logout();
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
        }
        return is;
    }

    /**
     * 下载文件存临时文件
     * @param ftpHost ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort ftp服务员器端口号
     * @param localPath 本地地址
     * @param ftpPath  ftp文件存放物理路径
     * @param fileName 文件路径
     */
    public static File downloadFile(String ftpHost, String ftpUserName,
                                    String ftpPassword, int ftpPort, String ftpPath, String localPath,
                                    String fileName) {
        FTPClient ftpClient = null;
        OutputStream os = null;
        //判断文件目录是否存在，如果不存在则创建
        File dir = new File(localPath);
        if(!dir.exists()) {
            dir.mkdirs();
        }
        File localFile = new File(localPath  + fileName);
        try {
            //连接ftp
            ftpClient = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            //设置编码
            ftpClient.setControlEncoding("UTF-8"); // 中文支持
            ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
            //设置linux ftp服务器
            FTPClientConfig conf = new FTPClientConfig( FTPClientConfig.SYST_UNIX);
            ftpClient.configure(conf);
            //设置访问被动模式
            ftpClient.setRemoteVerificationEnabled(false);
            ftpClient.enterLocalPassiveMode();
            //切换FTP目录
            ftpClient.changeWorkingDirectory(ftpPath);
            //获取文件流
            try(InputStream initialStream = ftpClient.retrieveFileStream(new String(fileName.getBytes("UTF-8"), "UTF-8"));
                OutputStream outStream = new FileOutputStream(localFile)) {
                byte[] buffer = new byte[8 * 1024];
                int bytesRead;
                while ((bytesRead = initialStream.read(buffer)) != -1) {
                    outStream.write(buffer, 0, bytesRead);
                }
            } catch (Exception e) {
                logger.error(e.getMessage(),e);
            }
        } catch (FileNotFoundException e) {
            logger.error("没有找到" + ftpPath + "文件");
            logger.error(e.getMessage(),e);
        } catch (SocketException e) {
            logger.error("连接FTP失败.");
            logger.error(e.getMessage(),e);
        } catch (IOException e) {
            logger.error("文件读取错误。");
            logger.error(e.getMessage(),e);
        }finally {
            try {
                if (os != null) os.close();
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
            try {
                if (ftpClient != null && ftpClient.isConnected()) {
                    ftpClient.logout();
                }
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
        }
        return localFile;
    }
    public static void main(String[] args) {
        //FtpUtils.downloadFile("192.168.43.202","anonymous","",21,"","d://","2019-01-01/1.txt");
    }

    /**
     * 上传文件
     * @param ftpHost ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort ftp服务员器端口号
     * @param ftpPath  ftp文件存放物理路径
     * @param fileName 文件路径
     * @param input 文件输入流，即从本地服务器读取文件的IO输入流
     */
    public static void uploadFile(String ftpHost, String ftpUserName,
                                  String ftpPassword, int ftpPort, String ftpPath,
                                  String fileName,InputStream input){
        FTPClient ftp=null;
        try {
            ftp=getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            ftp.changeWorkingDirectory(ftpPath);
            ftp.setFileType(FTP.BINARY_FILE_TYPE);
            fileName=new String(fileName.getBytes("GBK"),"iso-8859-1");
            ftp.enterLocalPassiveMode();
            boolean b = ftp.storeFile(fileName, input);
            System.out.println("1111"+b);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }finally {
            try {
                if (input != null){ input.close();}
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
            try {
                if (ftp != null) ftp.logout();
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
        }
    }

    public static boolean bcxUploadFile(String ftpHost, String ftpUserName,
                                        String ftpPassword, int ftpPort,
                                        String pathFile,String fileName ,InputStream input){

        BcxinFtpClient ftpClient = null;

        try {
            ftpClient = new BcxinFtpClient();
            ftpClient.connect(ftpHost, ftpUserName, ftpPassword, ftpPort);

            String ftpPath = pathFile.substring(0, pathFile.lastIndexOf("/")+1);

            boolean isDir = ftpClient.isARemoteDirectory(ftpPath);
            if (!isDir) {
                boolean createDirectoryFlag = ftpClient.createDirectory(ftpPath);
                if (!createDirectoryFlag) {
                    logger.error("目录创建失败：[{"+ftpPath+"}]");
                } else {
                    logger.debug("目录创建成功：[{"+ftpPath+"}]");
                }
            }
            String filePath = ftpPath + fileName;
            String fileDir = Paths.get(filePath).getParent().toString().replace("\\", "/");
            if (!ftpClient.isARemoteDirectory(fileDir)) {
                boolean createDirectoryFlag = ftpClient.createDirectory(fileDir);
                if (!createDirectoryFlag) {
                    logger.error("目录创建失败：[{"+ftpPath+"}]");
                } else {
                    logger.debug("目录创建成功：[{"+ftpPath+"}]");
                }
            }

            filePath = filePath.replace("\\", "/");

            return ftpClient.uploadFile(input, filePath);
        }catch (Exception e){
            logger.error(e.getMessage());
        }finally {
            try {
                if (input != null){ input.close();}
            }catch (Exception e){
                logger.error(e.getMessage(),e);
            }
            if(ftpClient != null){
                try {
                    ftpClient.disconnect();
                }catch (Exception e){
                    logger.error(e.getMessage());
                }
            }
        }
        return false;
    }

}