package com.bcxin.platform.service.company.impl;

import cn.hutool.core.text.StrSpliter;
import com.bcxin.platform.common.core.domain.entity.SysUser;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.PlatFormBusinessException;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.DateUtils;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.company.ComBaseInfo;
import com.bcxin.platform.domain.company.ComDepart;
import com.bcxin.platform.domain.company.PerBaseInfo;
import com.bcxin.platform.domain.company.PerOrgRelation;
import com.bcxin.platform.dto.PerBaseInfoDto;
import com.bcxin.platform.dto.attend.AttendPerDto;
import com.bcxin.platform.dto.company.PerInfoQueryDto;
import com.bcxin.platform.mapper.company.ComBaseInfoMapper;
import com.bcxin.platform.mapper.company.ComDepartMapper;
import com.bcxin.platform.mapper.company.PerBaseInfoMapper;
import com.bcxin.platform.mapper.company.PerOrgRelationMapper;
import com.bcxin.platform.service.company.PerBaseInfoService;
import com.bcxin.platform.mapper.system.SysUserMapper;
import com.bcxin.platform.util.PageInfoUtils;
import com.bcxin.platform.util.constants.CommonConst;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.util.StringUtil;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 人员基础信息Service业务层处理
 * 
 * @author task
 * @date 2021-04-09
 */
@Service
@Transactional("transactionManager")
public class PerBaseInfoServiceImpl implements PerBaseInfoService {

    @Resource
    private PerBaseInfoMapper perBaseInfoMapper;

    @Resource
    private ComBaseInfoMapper comBaseInfoMapper;

    @Resource
    private PerOrgRelationMapper perOrgRelationMapper;

    @Resource
    private ComDepartMapper comDepartMapper;

    @Resource
    private SysUserMapper sysUserMapper;

    @Resource
    private IdWorker idWorker;

    /**
     * 查询人员基础信息
     * 
     * @param perId 人员基础信息ID
     * @return 人员基础信息
     */
    @Override
    public PerBaseInfo selectPerBaseInfoById(Long perId){
        return perBaseInfoMapper.selectPerBaseInfoById(perId);
    }

    @Override
    public PerBaseInfo selectPerBaseInfoByTlkId(String tlkPerId){
        return perBaseInfoMapper.selectPerBaseInfoByTlkId(tlkPerId);
    }

    /**
     * 查询人员基础信息列表
     * 
     * @param perBaseInfo 人员基础信息
     * @return 人员基础信息
     */
    @Override
    public List<PerBaseInfo> selectPerBaseInfoList(PerBaseInfo perBaseInfo){
        return perBaseInfoMapper.selectPerBaseInfoList(perBaseInfo);
    }

    /**
     * 新增人员基础信息
     * 
     * @param perBaseInfo 人员基础信息
     * @return 结果
     */
    @Override
    public int insertPerBaseInfo(PerBaseInfo perBaseInfo){
        perBaseInfo.setPerId(idWorker.nextId());
        perBaseInfo.setCreateTime(DateUtils.getNowDate());
        return perBaseInfoMapper.insertPerBaseInfo(perBaseInfo);
    }

    /**
     * 新增修改人员基础信息
     * 
     * @param perBaseInfo 人员基础信息
     * @return 结果
     */
    @Override
    public int updatePerBaseInfo(PerBaseInfoDto perBaseInfo){
        int result = 0;

        SysUser user = sysUserMapper.selectUserByPhoneNumber(perBaseInfo.getMobilePhone());
        if(user == null){
            user = new SysUser();
            user.setUserId(idWorker.nextId());
            user.setUserName(perBaseInfo.getName());
            user.setPhonenumber(perBaseInfo.getMobilePhone());
            user.setLoginName(perBaseInfo.getMobilePhone());
            user.setCreateTime(DateUtils.getNowDate());
            user.setUpdateTime(DateUtils.getNowDate());
            sysUserMapper.insertUser(user);
        }else{
            if(StringUtils.isNotEmpty(perBaseInfo.getName()) && !perBaseInfo.getName().equals(user.getUserName())) {
                user.setUpdateTime(DateUtils.getNowDate());
                user.setUserName(perBaseInfo.getName());
                sysUserMapper.updateUser(user);
            }
        }

        perBaseInfo.setUserId(user.getUserId());

        PerBaseInfo dbPerBaseInfo = perBaseInfoMapper.selectPerBaseInfoByTlkId(perBaseInfo.getTlkPerId());

        if(StringUtils.isNotEmpty(perBaseInfo.getTlkComId())){
            ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoByTlkId(perBaseInfo.getTlkComId());
            if(comBaseInfo == null){
                throw new PlatFormBusinessException("公司不存在");
            }
            perBaseInfo.setComId(comBaseInfo.getComId());
        }

        if(dbPerBaseInfo == null){
            perBaseInfo.setPerId(idWorker.nextId());
            perBaseInfo.setCreateTime(DateUtils.getNowDate());
            result = perBaseInfoMapper.insertPerBaseInfo(perBaseInfo);
        }else {
            perBaseInfo.setIsDelete(CommonConst.N);
            perBaseInfo.setUpdateTime(DateUtils.getNowDate());
            result = perBaseInfoMapper.updatePerBaseInfo(perBaseInfo);
        }


        if(StringUtils.isNotEmpty(perBaseInfo.getTlkDepartIds())){
            String[] tlkDepartIds = perBaseInfo.getTlkDepartIds().split(",");
            List<ComDepart> comDeparts = comDepartMapper.selectComDepartByTlkIds(tlkDepartIds);
            Map<String, ComDepart> comDepartMap = comDeparts.stream().collect(Collectors.toMap(ComDepart::getTlkDepartId, Function.identity()));

            List<PerOrgRelation> relations = new ArrayList<>();
            PerOrgRelation relation = null;
            ComDepart comDepart = null;
            for (String tlkDepartId : tlkDepartIds) {
                comDepart = comDepartMap.get(tlkDepartId);
                if(comDepart == null){
                    throw new PlatFormBusinessException(tlkDepartId + "部门不存在");
                }
                relation = new PerOrgRelation();
                relation.setComId(perBaseInfo.getComId());
                relation.setOrgId(comDepart.getDepartId());
                relation.setPerId(perBaseInfo.getPerId());
                PerOrgRelation dbPerOrgRelation = perOrgRelationMapper.find(relation);
                if (dbPerOrgRelation == null) {
                    relation.setPerOrgRelationId(idWorker.nextId());
                    relation.setCreateTime(DateUtils.getNowDate());
                    relation.setUpdateTime(DateUtils.getNowDate());
                    relations.add(relation);
                }else if(CommonConst.DELETE.equals(dbPerOrgRelation.getIsDelete())){
                    dbPerOrgRelation.setUpdateTime(DateUtils.getNowDate());
                    dbPerOrgRelation.setIsDelete(CommonConst.N);
                    relations.add(dbPerOrgRelation);
                }
            }
            perOrgRelationMapper.saveBatch(relations);

        }
        return result;
    }


    /**
     * 新增修改人员基础信息
     *
     * @param list 人员
     * @return 结果
     */
    @Override
    public int editBatch(List<PerBaseInfoDto> list){
        for (PerBaseInfoDto perBaseInfoDto : list) {
            updatePerBaseInfo(perBaseInfoDto);
        }
        return list.size();
    }

    /**
     * 删除人员基础信息对象
     * 
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deletePerBaseInfoByIds(String ids){
        return perBaseInfoMapper.deletePerBaseInfoByIds(Convert.toStrArray(ids));
    }

    /**
     * 删除人员基础信息信息
     * 
     * @param perId 人员基础信息ID
     * @return 结果
     */
    @Override
    public int deletePerBaseInfoById(Long perId){
        return perBaseInfoMapper.deletePerBaseInfoById(perId);
    }

    /**
     * 通过人员ID列表查询人员基础信息
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-04-15
     */
    @Override
    public Result getPerBaseInfoListByPerIds(PerBaseInfoDto dto) {
        if (StringUtil.isEmpty(dto.getPerIds())) {
            return Result.fail("人员ID不能为空！");
        }
        List<String> perIdList = StrSpliter.split(dto.getPerIds(), CommonConst.COMMA, true, true);
        List<Map<String, String>> perList = perOrgRelationMapper.getPerBaseInfoListByPerIds(dto.getComId(), perIdList);
        return Result.success(Result.SUCCESS_QUERY_MSG, perList);
    }

    /**
     * 华为云会议人员信息查询
     *
     * @param dto
     * @return
     * @author llc
     * @date 2019-12-11
     */
    @Override
    public Result pageForhuaweiMeetPerInfo(PerInfoQueryDto dto) {
        if (dto.getPageNumber() == null) {
            throw new V5BusinessException(Result.ERROR, "第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            throw new V5BusinessException(Result.ERROR, "每页多少行不能为空");
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        List<LinkedHashMap<String, Object>> list = perBaseInfoMapper.pageForhuaweiMeetPerInfo(dto);
        return Result.success(Result.SUCCESS_QUERY_MSG, new PageInfoUtils(list));
    }

    @Override
    public void saveBatch(List<PerBaseInfo> perBaseInfoList) {
        perBaseInfoMapper.saveBatch(perBaseInfoList);
    }

    @Override
    public List<PerBaseInfo> selectPerBaseInfoByTlkIds(List<String> tlkIds) {
        return perBaseInfoMapper.selectPerBaseInfoByTlkIds(tlkIds);
    }

    @Override
    public List<PerBaseInfo> selectPerBaseInfoByPerIds(AttendPerDto searchDto) {
        return perBaseInfoMapper.selectPerBaseInfoByPerIds(searchDto);
    }
}
