package com.bcxin.oa.old.service.salary.web;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.StrSplitter;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelWriter;
import com.bcxin.oa.old.common.utils.DateUtils;
import com.bcxin.oa.old.common.utils.ExcelUtil;
import com.bcxin.oa.old.common.utils.IdWorker;
import com.bcxin.oa.old.common.utils.PageInfoUtils;
import com.bcxin.oa.old.common.utils.file.BcxinFileUtils;
import com.bcxin.oa.old.mapper.PerBaseInfoMapper;
import com.bcxin.oa.old.mapper.attend.CountPerAttendDateMapper;
import com.bcxin.oa.old.mapper.salary.ComSalaryRecordMapper;
import com.bcxin.oa.old.mapper.salary.ConfigTaxMapper;
import com.bcxin.oa.old.mapper.salary.PerSalaryGrantDetailMapper;
import com.bcxin.oa.old.mapper.salary.PerSalaryGrantMapper;
import com.bcxin.oa.old.mapper.task.CountPerTaskDateMapper;
import com.bcxin.oa.old.entity.salary.ComSalaryRecord;
import com.bcxin.oa.old.entity.salary.ConfigTax;
import com.bcxin.oa.old.entity.salary.PerSalaryGrant;
import com.bcxin.oa.old.entity.salary.PerSalaryGrantDetail;
import com.bcxin.oa.old.dto.ExportPerAttendMonthDTO;
import com.bcxin.oa.old.dto.ParamDTO;
import com.bcxin.oa.old.dto.salary.ExportPerSalaryMonthDTO;
import com.bcxin.oa.old.dto.salary.PerSalaryParamDTO;
import com.bcxin.oa.old.common.exception.BusinessException;
import com.bcxin.oa.old.service.common.CommonService;
import com.bcxin.oa.old.service.system.CacheService;
import com.bcxin.oa.old.common.*;
import com.bcxin.oa.old.common.CommonConst;
import com.bcxin.oa.old.common.DictConst;
import com.bcxin.oa.old.common.DictMessageTypeConst;
import com.bcxin.oa.old.common.MsgConst;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.StringUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.util.CellRangeAddress;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 企业薪酬模板
 *
 * @author llc
 * @date 2018-12-28
 */
@Service
@Transactional
public class PerSalaryGrantServiceImpl implements PerSalaryGrantService {

    /**
     * 税收起征点
     */
    private final static BigDecimal THRESHOLD = new BigDecimal("5000");

    /**
     * 最大excel导入数量
     */
    private final static int MAX_UPLOAD_SIZE = 503;

    /**
     * 标题行行数
     */
    private final static int MIN_UPLOAD_SIZE = 3;

    @Resource
    private CacheService cacheService;

    @Resource
    private CommonService commonService;

    @Resource
    private PerBaseInfoMapper perBaseInfoMapper;

    @Resource
    private PerSalaryGrantMapper perSalaryGrantMapper;

    @Resource
    private PerSalaryGrantDetailMapper perSalaryGrantDetailMapper;

    @Resource
    private CountPerAttendDateMapper countPerAttendDateMapper;

    @Resource
    private CountPerTaskDateMapper countPerTaskDateMapper;

    @Resource
    private ConfigTaxMapper configTaxMapper;

    @Resource
    private ComSalaryRecordMapper comSalaryRecordMapper;

    @Resource
    private IdWorker idWorker;
    /**
     * 核算薪资
     * @param perSalaryGrant
     * @return
     * @author llc
     * @date 2019-01-02
     */
    @Override
    public Result calSalary(PerSalaryGrant perSalaryGrant) throws BusinessException{

        /*** 由于某些企业人员比较多可能到达1W人、如果在java层处理业务逻辑需要频繁从数据库读取数据并计算更新
         *   且计算时会存在多个循环嵌套，及其占内存，考虑到耗时、资源问题，业务逻辑不建议java代码写
         *   1.通过企业薪酬模板选择的人员批量插入到人员薪资发放表
         *   2.通过企业薪酬模板选择的项目批量插入到人员薪资发放明细表
         *   3.通过医社保记录批量更新医社保、公积金记录
         *   4.通过考勤报表批量更新考勤薪资记录
         *   5.更新未安排考勤的人员考勤薪资为0
         *   6.通过外勤报表批量更新外勤薪资记录
         *   7.更新未安排外勤的人员外勤薪资为0
         *   8.计算个人所得税
         *   9.更新各项工资
         */

         if (perSalaryGrant.getComId() == null) {
             return Result.fail("企业Id不能为空");
         }
         if (StringUtil.isEmpty(perSalaryGrant.getGrantYears())) {
             return Result.fail("核算年月不能为空");
         }

         /*** 查询出薪资模板选择的人员（排除当月工资单已经确认的员工）***/
         List<PerSalaryGrant> perSalaryGrantList = perSalaryGrantMapper.getSalaryTempPerList(perSalaryGrant.getComId(),perSalaryGrant.getGrantYears());
         if (perSalaryGrantList.size() == 0) {
            return Result.fail("没有参与核算人员，请在薪资模板设置");
         }

         for(PerSalaryGrant salaryGrant : perSalaryGrantList){
             salaryGrant.setPerSalaryGrantId(idWorker.nextId());
         }

         /*** 重新核算前需要清除员工未确认的记录 ***/
        perSalaryGrantDetailMapper.deletePerNoSureGrantDetail(perSalaryGrant.getComId(),perSalaryGrant.getGrantYears());
        perSalaryGrantMapper.deletePerNoSureGrant(perSalaryGrant.getComId(),perSalaryGrant.getGrantYears());

         /*** 1.通过企业薪酬模板选择的人员批量插入到人员薪资发放表 ***/
         perSalaryGrantMapper.batchInsertPerSalaryGrant(perSalaryGrant.getComId(),
                                                           perSalaryGrant.getGrantYears(),
                                                           perSalaryGrant.getCreateBy(),
                                                           perSalaryGrantList);

        /*** 2.通过企业薪酬模板选择的项目批量插入到人员薪资发放明细表 ***/
        perSalaryGrantDetailMapper.batchInsertPerSalaryGrantDetail(perSalaryGrant.getComId(),
                                                                      perSalaryGrant.getGrantYears(),
                                                                      perSalaryGrant.getCreateBy());

        /*** 3.通过医社保记录批量更新医社保、公积金记录 ***/
        perSalaryGrantDetailMapper.batchUpdatePerSocialIns(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 4.通过考勤报表批量更新考勤薪资记录 ***/
        perSalaryGrantDetailMapper.batchUpdatePerAttendSalary(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 5.更新没有安排考勤的考勤薪资为0  ***/
        perSalaryGrantDetailMapper.batchUpdateNoAttendPerAttendSalary(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 6.通过外勤报表批量更新外勤薪资记录 ***/
        perSalaryGrantDetailMapper.batchUpdatePerTaskSalary(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 7.更新没有安排外勤的考勤外勤为0  ***/
        perSalaryGrantDetailMapper.batchUpdateNoTaskPerTaskSalary(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 8.计算个人所得税  ***/
        perSalaryGrantDetailMapper.batchCalPerTax(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 9.更新各项工资  ***/
        perSalaryGrantMapper.batchUpdatePerSalary(perSalaryGrant.getComId(), perSalaryGrant.getGrantYears());

        /*** 10.生成企业薪资记录  ***/

        ComSalaryRecord comSalaryRecord = new ComSalaryRecord();
        comSalaryRecord.setComSalaryRecordId(idWorker.nextId());
        comSalaryRecord.setCreateTime(perSalaryGrant.getCreateTime());
        comSalaryRecord.setCreateBy(perSalaryGrant.getCreateBy());
        comSalaryRecord.setGrantYears(perSalaryGrant.getGrantYears());
        comSalaryRecord.setIsCreateData(CommonConst.N);
        comSalaryRecord.setComId(perSalaryGrant.getComId());
        comSalaryRecordMapper.insertComSalaryRecord(comSalaryRecord);

        return  Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 校验是否核算
     *
     * @param dto
     * @return
     * @author llc
     * @date 2019-01-11
     */

    @Override
    public Result checkIsAccount(PerSalaryParamDTO dto) throws BusinessException {
        int count = perSalaryGrantMapper.getSalaryCountByYears(dto);
        String existsFlag = CommonConst.Y;
        if(count == 0){
            existsFlag = CommonConst.N;
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, existsFlag);
    }


    /**
     * 人员月度工资列表
     *
     * @param dto
     * @return
     * @author llc
     * @date 2019-01-03
     */

    @Override
    public Result getPerSalaryList(PerSalaryParamDTO dto) throws BusinessException {
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        List<Map<String, Object>> list = perSalaryGrantMapper.getPerSalaryList(dto);
        PageInfoUtils pageInfo = new PageInfoUtils(list);
        return Result.success(Result.SUCCESS_QUERY_MSG, pageInfo);
    }



    /**
     * 个人工资单查询
     * @return
     * @author 罗鹏
     * @date 2019-01-01
     */
    @Override
    public Result getPerSalaryDetail(ParamDTO paramDTO) throws BusinessException{
        if ( paramDTO.getId() == null ) {
            throw new BusinessException(Result.ERROR, "薪资id不能为空");
        }
        Map<String,Object> resultMap = Maps.newHashMap();
        /* 获得个人工资条 */
        PerSalaryGrant salaryGrant = perSalaryGrantMapper.getByPrimaryKey(Long.parseLong(paramDTO.getId()));
        /* 获得员工个人信息 */
        Map perMap = perBaseInfoMapper.selectPerDetail(salaryGrant.getPerId(), salaryGrant.getComId());
        /* 获得个人工资条动态的明细 */
        List<Map> dynamicMap = perSalaryGrantDetailMapper.getDynamicBySalaryId(salaryGrant.getPerSalaryGrantId());
        List<Map> fixMap = perSalaryGrantDetailMapper.getFixBySalaryId(salaryGrant.getPerSalaryGrantId());
        Map<String,Object> map1 = Maps.newHashMap();
        map1.put("salaryType","sumAmount");
        map1.put("salaryName","税前工资");
        map1.put("amount",salaryGrant.getSumAmount());

        Map<String,Object> map2 = Maps.newHashMap();
        map2.put("salaryType","taxableAmount");
        map2.put("salaryName","计税工资");
        map2.put("amount",salaryGrant.getTaxableAmount());

        Map<String,Object> map3 = Maps.newHashMap();
        map3.put("salaryType","payableAmount");
        map3.put("salaryName","应发工资");
        map3.put("amount",salaryGrant.getPayableAmount());

        fixMap.add(map1);
        fixMap.add(map2);
        fixMap.add(map3);

        /* 获得考勤明细 */
        paramDTO.setPerId(salaryGrant.getPerId());
        Date startDate = DateUtil.parseDate(salaryGrant.getGrantYears()+"-01");
        paramDTO.setStartDate(salaryGrant.getGrantYears()+"-01");
        paramDTO.setEndDate(DateUtil.formatDate(DateUtil.endOfMonth(startDate)));
        ExportPerAttendMonthDTO attendDetail = countPerAttendDateMapper.getPerAttendMonthCount(paramDTO);
        ExportPerAttendMonthDTO taskAttendDetail = countPerTaskDateMapper.getPerTaskAttendMonthCount(paramDTO);
        resultMap.put("perMap",perMap);
        resultMap.put("salaryGrant",salaryGrant);
        resultMap.put("dynamicMap",dynamicMap);
        resultMap.put("fixMap",fixMap);
        resultMap.put("attendDetail",attendDetail);
        resultMap.put("taskAttendDetail",taskAttendDetail);
        return Result.success(Result.SUCCESS_QUERY_MSG,resultMap);
    }


    /**
     * 批量核算工资条
     * @return
     */
    @Override
    public Result batchAccountSalary(ParamDTO paramDTO)  throws BusinessException{
        if (StringUtil.isEmpty(paramDTO.getIds())) {
            throw new BusinessException(Result.ERROR,"请选择记录");
        }
        List<String> idList = StrSplitter.split(paramDTO.getIds(), ",", true, true);
        perSalaryGrantMapper.batchAccountSalary(idList);

        List<PerSalaryGrant> salaryList = perSalaryGrantMapper.getByIdList(idList);

        for (PerSalaryGrant grant:salaryList) {
            String content = MsgConst.TYPE_030402_SALARY_PER_CONFIRM_CONTENT;
            content =  content.replace("{month}",grant.getGrantYears());
            /* 生成通知 */
            commonService.sendMessageToApp(MsgConst.TYPE_030402_SALARY_PER_CONFIRM_TITLE, content, DictMessageTypeConst.MESSAGETYPE_030402, String.valueOf(grant.getPerId()),
                    paramDTO.getComId(), grant.getPerSalaryGrantId());
        }
        return  Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 删除工资条
     * @return
     */
    @Override
    public Result deleteSalaryGrant(ParamDTO paramDTO){
        if (StringUtil.isEmpty(paramDTO.getId())) {
            throw new BusinessException(Result.ERROR,"请选择记录");
        }
        perSalaryGrantDetailMapper.deleteByGrantId(Long.parseLong(paramDTO.getId()));
        perSalaryGrantMapper.deleteByPrimaryKey(Long.parseLong(paramDTO.getId()));
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 个人工资条修改
     * @return
     * @author lp
     * @date 2019-01-01
     */
    @Override
    public Result updateSalaryDetail(Map<String,String> paramMap)  throws BusinessException{
        /* 修改工资条的详细信息 */
        String perSalaryGrantId = paramMap.get("perSalaryGrantId");
        if (StringUtil.isEmpty(perSalaryGrantId)) {
            throw new BusinessException(Result.ERROR, "薪资id不能为空");
        }
        PerSalaryGrant salaryGrant = perSalaryGrantMapper.getByPrimaryKey(Long.parseLong(perSalaryGrantId));
        /* 计算个税 */
        Map<String, BigDecimal> resultTaxMap = calcTax(paramMap, salaryGrant);
        perSalaryGrantDetailMapper.updateLine(salaryGrant.getPerSalaryGrantId()
                ,DictConst.SALARYTYPE_DETAIL_GRSDS,resultTaxMap.get("tax").toString());

        /* 修改工资条明细 */
        List<PerSalaryGrantDetail> detailList = Lists.newArrayList();
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_JBGZ);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_JXGZ);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_GWBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_CLBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_TXBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_CYBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_QTBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_CDKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_ZTKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_KGKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_SJKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_BJKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_LKKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQCDKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQZTKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQKGKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQSJKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQBJKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_WQLKKK);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_GZRJBBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_XXRJBBT);
        packLine(detailList,paramMap,DictConst.SALARYTYPE_DETAIL_JJRJBBT);

        /* 批量修改 */
        perSalaryGrantDetailMapper.updateLineList(Long.parseLong(perSalaryGrantId),detailList);

        /* 修改税前，税后，应发工资 */
        salaryGrant.setPayableAmount(resultTaxMap.get("payableAmount"));
        salaryGrant.setTaxableAmount(resultTaxMap.get("taxableAmount"));
        salaryGrant.setSumAmount(resultTaxMap.get("sumAmount"));
        perSalaryGrantMapper.update(salaryGrant);

        return  Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 修改项
     * @param paramMap
     * @param salaryType
     */
    private void packLine(List<PerSalaryGrantDetail> detailList,Map<String,String> paramMap,String salaryType){
        String amount = paramMap.get(salaryType);
        PerSalaryGrantDetail detail = new PerSalaryGrantDetail();
        if (StringUtil.isEmpty(amount)) {
            detail.setAmount(BigDecimal.ZERO);
        } else {
            detail.setAmount(new BigDecimal(amount));
        }
        detail.setSalaryType(salaryType);
        detailList.add(detail);
    }

    /**
     * 计算个税
     * 2019年最新计税公式，5000起征点
     * 1.应纳税所得额 = 税前工资收入金额 － 五险一金(个人缴纳部分) －费用减除额
     * 2.应纳税额 = 应纳税所得额 x 税率 － 速算扣除数
     * @return
     */
    private Map<String,BigDecimal> calcTax(Map<String,String> paramMap,PerSalaryGrant salaryGrant){
        Map<String,BigDecimal> resultMap = Maps.newHashMap();
        BigDecimal jbgz = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_JBGZ);
        BigDecimal jxgz = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_JXGZ);
        BigDecimal gwbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_GWBT);
        BigDecimal clbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_CLBT);
        BigDecimal txbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_TXBT);
        BigDecimal cybt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_CYBT);
        BigDecimal qtbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_QTBT);
        BigDecimal cdkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_CDKK);
        BigDecimal ztkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_ZTKK);
        BigDecimal kgkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_KGKK);
        BigDecimal sjkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_SJKK);
        BigDecimal bjkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_BJKK);
        BigDecimal lkkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_LKKK);
        BigDecimal wqcdkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQCDKK);
        BigDecimal wqztkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQZTKK);
        BigDecimal wqkgkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQKGKK);
        BigDecimal wqsjkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQSJKK);
        BigDecimal wqbjkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQBJKK);
        BigDecimal wqlkkk = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_WQLKKK);
        BigDecimal gzrjbbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_GZRJBBT);
        BigDecimal xxrjbbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_XXRJBBT);
        BigDecimal jjrjbbt = safeBigDecimal(paramMap,DictConst.SALARYTYPE_DETAIL_JJRJBBT);

        BigDecimal amount = NumberUtil.add(jbgz,jxgz,gwbt,clbt,txbt,cybt,qtbt,gzrjbbt,xxrjbbt,jjrjbbt);
        BigDecimal subAmount = NumberUtil.add(cdkk,ztkk,kgkk,sjkk,bjkk,lkkk,wqcdkk,wqztkk,wqkgkk,wqkgkk,wqsjkk,wqbjkk,wqlkkk);
        /* 得到税前工资收入金额 */
        BigDecimal taxBeforeAmount = amount.subtract(subAmount);
        /* 得到五险一金 */
        List<Map> detailMap = perSalaryGrantDetailMapper.getFixBySalaryId(salaryGrant.getPerSalaryGrantId());
        BigDecimal dkylj = BigDecimal.ZERO;
        BigDecimal dkyb = BigDecimal.ZERO;
        BigDecimal dkgjj = BigDecimal.ZERO;
        BigDecimal dksyj = BigDecimal.ZERO;
        for (Map map :detailMap) {
            if (DictConst.SALARYTYPE_DETAIL_DKYLJ.equals(map.get("salaryType"))) {
                dkylj = new BigDecimal(map.get("amount").toString());
            } else if (DictConst.SALARYTYPE_DETAIL_DKYB.equals(map.get("salaryType"))) {
                dkyb = new BigDecimal(map.get("amount").toString());
            } else if (DictConst.SALARYTYPE_DETAIL_DKGJJ.equals(map.get("salaryType"))) {
                dkgjj = new BigDecimal(map.get("amount").toString());
            } else if (DictConst.SALARYTYPE_DETAIL_DKSYJ.equals(map.get("salaryType"))) {
                dksyj = new BigDecimal(map.get("amount").toString());
            }
        }
        /* 得到应纳税所得额 */
        BigDecimal taxAmount = taxBeforeAmount.subtract(NumberUtil.add(dkylj,dkyb,dkgjj,dksyj));
        /* 得到超出起征点的部分 */
        BigDecimal aboveAmount = taxAmount.subtract(THRESHOLD);
        BigDecimal tax = BigDecimal.ZERO;
        /* 应税工资大于0才进行税费计算 */
        if (aboveAmount.compareTo(BigDecimal.ZERO) > 0) {
            ConfigTax configTax = configTaxMapper.getTax(aboveAmount.intValue());
            /* 获得税额 */
            tax = aboveAmount.multiply(new BigDecimal(configTax.getTaxRate()))
                    .divide(new BigDecimal(100),2,BigDecimal.ROUND_DOWN)
                    .subtract(new BigDecimal(configTax.getDeduction()));
        }

        /** 工资总额(不含代扣五险、公积金、个税) */
        resultMap.put("sumAmount",taxBeforeAmount);
        /** 应税工资(工资总额-五险-公积金) */
        resultMap.put("taxableAmount",taxAmount);
        /** 应发金额=工资总额-代扣(五险、公积金、个税) */
        resultMap.put("payableAmount",taxAmount.subtract(tax));
        /* 个税 */
        resultMap.put("tax",tax);
        return resultMap;
    }

    private BigDecimal safeBigDecimal(Map<String,String> paramMap,String salaryType){
        String amount = paramMap.get(salaryType);
        if (StringUtil.isEmpty(amount)) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(amount);
    }

    /**
     * 人员月度工资导出
     * @param dto
     * @return
     * @author llc
     * @date 2019-01-17
     */
    @Override
    public Result exportPerMonthSalary(PerSalaryParamDTO dto, HttpServletResponse response)
            throws BusinessException {
        if (StringUtils.isEmpty(dto.getGrantYears())) {
            throw new BusinessException(Result.ERROR, "月份不能为空");
        }
        OutputStream os = null;
        HSSFWorkbook wb = new HSSFWorkbook();
        try {
            // String comName =
            // countPerAttendDateMapper.getComNameByComId(dto.getComId());
            String fileName = "_" + dto.getGrantYears().replace("-","") + "工资" + DateUtils.getDate("yyyyMMddHHmmss") + ".xls";
            response.setContentType("application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;fileName=" + new String(fileName.getBytes("gb2312"), "ISO8859-1"));
            os = response.getOutputStream();
            HSSFSheet sheet1 = wb.createSheet("工资");

            List<ExportPerSalaryMonthDTO> list = perSalaryGrantMapper.getPerSalaryDetailList(dto);

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.HSSFColorPredefined.BLUE.getIndex());
            titleFont.setFontName("黑体");
            titleStyle.setAlignment(HorizontalAlignment.CENTER);// 水平居中
            titleStyle.setVerticalAlignment(VerticalAlignment.CENTER);// 垂直居中
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            HSSFDataFormat format = wb.createDataFormat();
            titleStyle.setDataFormat(format.getFormat("@"));

            /*** 合并单元格 ***/

            for (int i = 0; i <= 4; i++) {
                sheet1.addMergedRegion(new CellRangeAddress(0, (short) 1, i, i));
            }
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 5, 6)); // 员工工资
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 7, 11)); // 福利补贴
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 12, 20));// 内勤计薪
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 21, 24)); // 社保公积金

            /*** 填充第一行数据 ***/
            HSSFRow rowFirst = sheet1.createRow(0);
            String[] firstArr = {"序号", "姓名","人员类别","部门","应发工资", "基本工资(+)", "绩效工资(+)", "岗位津贴(+)", "差旅补贴(+)", "通讯补贴(+)", "餐饮补贴(+)", "其他补贴(+)",
                                 "迟到扣款(-)", "早退扣款(-)", "旷工扣款(-)", "事假扣款(-)", "病假扣款(-)", "工作日加班补贴(+)", "休息日加班补贴(+)", "节假日加班补贴(+)", "漏卡扣款(-)",
                                 "代扣养老金(-)", "代扣失业金(-)", "代扣医保(-)", "代扣公积金(-)", "个人所得税(-)"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第二行的每个单元格
                if (i == 0) {
                    column = 1500;
                } else if (i == 2 || (i >= 4 && i <= 16) || i == 20) {
                    column = 3500;
                } else if (i == 3 || (i >= 17 && i <= 19)) {
                    column = 6000;
                }else if (i >= 21) {
                    column = 4000;
                }  else {
                    column = 2200;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
                cell.setCellStyle(titleStyle); // 文本格式
            }

            String[] arr = {"序号", "姓名", "人员类别", "部门", "应发工资", "员工工资", "福利补贴", "内勤计薪", "社保公积金", "个人所得税"};

            for (int i = 0; i < arr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第二行的每个单元格
                cell.setCellValue(arr[i]);
                cell.setCellStyle(titleStyle);
            }

            /*** 填充第一行数据 ***/
            ;
            HSSFRow rowSecond = sheet1.createRow(1);
            String[] secondArr = {"基本工资(+)", "绩效工资(+)", "岗位津贴(+)", "差旅补贴(+)", "通讯补贴(+)", "餐饮补贴(+)", "其他补贴(+)", "迟到扣款(-)", "早退扣款(-)", "旷工扣款(-)", "事假扣款(-)",
                                  "病假扣款(-)", "工作日加班补贴(+)","休息日加班补贴(+)","节假日加班补贴(+)","漏卡扣款(-)","代扣养老金(-)","代扣失业金(-)","代扣医保(-)","代扣公积金(-)","个人所得税(-)"};
            for (int i = 0; i < secondArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i + 5); // 获取第二行的每个单元格
                cell.setCellValue(secondArr[i]);
                cell.setCellStyle(titleStyle);
            }
            rowFirst.setHeight((short) 500);


            HSSFCell cell7 = rowFirst.createCell(7);
            cell7.setCellValue("福利补贴");
            cell7.setCellStyle(titleStyle);

            HSSFCell cell12 = rowFirst.createCell(12);
            cell12.setCellValue("内勤计薪");
            cell12.setCellStyle(titleStyle);

            HSSFCell cell21 = rowFirst.createCell(21);
            cell21.setCellValue("社保公积金");
            cell21.setCellStyle(titleStyle);

            HSSFCell cell25 = rowFirst.createCell(25);
            cell25.setCellValue("个人所得税");
            cell25.setCellStyle(titleStyle);


            /*** 写第一行 ***/

            /**** 填充数据 ***/

            /*** 字体橙色样式 ***/
            CellStyle orangeStyle = ExcelUtil.getOrangeStyle(wb);
            HSSFRow row = null;
            int rowIndex = 2;
            for (ExportPerSalaryMonthDTO result : list) {
                row = sheet1.createRow(rowIndex);
                Field[] fields = result.getClass().getDeclaredFields(); // 反射获取值
                int cellIndex = 0;
                for (Field f : fields) {
                    f.setAccessible(true); // 可以访问private对象不抛异常
                    String filedCode = f.getName();
                    Object value = ReflectUtil.getFieldValue(result, filedCode);
                    HSSFCell cell = row.createCell(cellIndex);
                    cell.setCellValue(value != null ? value.toString() : null);
                    if (filedCode.equalsIgnoreCase("payableAmount")) {
                            cell.setCellStyle(orangeStyle);
                    }
                    cellIndex++;
                }
                rowIndex++;
            }
            wb.write(os);
            os.close();
        } catch (Exception e) {
            throw new BusinessException(Result.ERROR, e.getMessage());
        } finally {
            try {
                os.close();
            } catch (IOException e1) {
            }
        }
        return Result.success("导出月度工资失败");
    }

    /**
     * 导出人员薪酬模板
     * @return
     */
    @Override
    public Result exportSalaryTemplate(){
        String url = cacheService.getSystemConfig("MATERIAL_SALARY_TEMPLATE_URL");
        return Result.success(CommonConst.OP_SUCCESS,url);
    }
    /**
     * 导入excel薪酬报表数据
     *
     * @param paramDTO
     * @return
     */
    @Override
    public Result importSalaryReportInfo(MultipartFile file, ParamDTO paramDTO) {
        Long comId = paramDTO.getComId();
        String reportMonth = paramDTO.getMonth();

        if (comId == null) {
            throw new BusinessException(Result.ERROR, "公司id不能为空");
        }
        if (StringUtils.isEmpty(reportMonth)) {
            throw new BusinessException(Result.ERROR, "月份不能为空");
        }

        try (InputStream in = file.getInputStream()){
			/* 读取excel，支持 */
            ExcelReader reader = cn.hutool.poi.excel.ExcelUtil.getReader(in);
            List<List<Object>> xlsList = reader.read();

            if (CollectionUtil.isEmpty(xlsList)) {
                return Result.fail(PublicConst.OP_IMPORT_FAILURE, buildFailExcel(Lists.newArrayList("无效的excel，未检测到数据")));
            }
            /* 仅仅只有标题行 */
            if ( xlsList.size() == MIN_UPLOAD_SIZE ) {
                return Result.fail(PublicConst.OP_IMPORT_FAILURE, buildFailExcel(Lists.newArrayList("请导入薪酬数据，未检测到数据")));
            }
            if ( xlsList.size() > MAX_UPLOAD_SIZE ) {
                return Result.fail(PublicConst.OP_IMPORT_FAILURE, buildFailExcel(Lists.newArrayList("最大导入数量为500条")));
            }
            /* 拿到第三行，为标题行*/
            List<Object> titleList = xlsList.get(2);
		    /* 如果模板的列与导入的列不一致，说明不是下载的标准模板 */
            if (titleList.size() != 32) {
                return Result.fail(PublicConst.OP_IMPORT_FAILURE, buildFailExcel(Lists.newArrayList("excel列不符，请重新下载导入模板")));
            }

            /* 初始化一些必要的集合，供对比插入用 */
            List<String> validPerIdList = Lists.newArrayList();
            List<Map<String,String>> perMapList = perBaseInfoMapper.selectNameKeyByComId(comId);
            List<String> perKeyList = Lists.newArrayList();
            Map<String,String> perKeyMap = Maps.newHashMap();
            for (Map<String,String> map:perMapList) {
                perKeyList.add(map.get("nameKey"));
                perKeyMap.put(map.get("nameKey"),map.get("perId"));
            }

            /* 得到必填字段Map */
            Map<Integer,String> requiredMap = Maps.newHashMap();
            requiredMap.put(0,"姓名");
            requiredMap.put(1,"身份证");

		    /* 得到需要校验数字的字段 */
            Map<Integer, String> digitMap = Maps.newHashMap();
            digitMap.put(2,"基本工资");
            digitMap.put(3,"绩效工资");
            digitMap.put(4,"岗位津贴");
            digitMap.put(5,"差旅补贴");
            digitMap.put(6,"通讯补贴");
            digitMap.put(7,"餐饮补贴");
            digitMap.put(8,"其他补贴");
            digitMap.put(9,"迟到扣款");
            digitMap.put(10,"早退扣款");
            digitMap.put(11,"旷工扣款");
            digitMap.put(12,"事假扣款");
            digitMap.put(13,"病假扣款");
            digitMap.put(14,"漏卡扣款");
            digitMap.put(15,"迟到扣款");
            digitMap.put(16,"早退扣款");
            digitMap.put(17,"旷工扣款");
            digitMap.put(18,"事假扣款");
            digitMap.put(19,"病假扣款");
            digitMap.put(20,"漏卡扣款");
            digitMap.put(21,"工作日加班补贴");
            digitMap.put(22,"休息日加班补贴");
            digitMap.put(23,"节假日加班补贴");
            digitMap.put(24,"代扣养老金");
            digitMap.put(25,"代扣失业金");
            digitMap.put(26,"代扣医保");
            digitMap.put(27,"代扣公积金");
            digitMap.put(28,"个人所得税");
            digitMap.put(29,"税前工资");
            digitMap.put(30,"计税工资");
            digitMap.put(31,"应发工资");

            /* 校验excel的数据合规性 */
            List<String> errorList = validateReportInfo(xlsList,requiredMap,digitMap,validPerIdList,perKeyList,perKeyMap);
            if (CollectionUtil.isNotEmpty(errorList)) {
                /* 如果有失败信息,生成失败的人员信息文件 ***/
                return Result.fail(PublicConst.OP_IMPORT_FAILURE, buildFailExcel(errorList));
            }

            /* 初始化薪酬项与excel列对应 */
            Map<Integer,String> salaryMap = Maps.newHashMap();
            salaryMap.put(2,DictConst.SALARYTYPE_DETAIL_JBGZ);
            salaryMap.put(3,DictConst.SALARYTYPE_DETAIL_JXGZ);
            salaryMap.put(4,DictConst.SALARYTYPE_DETAIL_GWBT);
            salaryMap.put(5,DictConst.SALARYTYPE_DETAIL_CLBT);
            salaryMap.put(6,DictConst.SALARYTYPE_DETAIL_TXBT);
            salaryMap.put(7,DictConst.SALARYTYPE_DETAIL_CYBT);
            salaryMap.put(8,DictConst.SALARYTYPE_DETAIL_QTBT);
            salaryMap.put(9,DictConst.SALARYTYPE_DETAIL_WQCDKK);
            salaryMap.put(10,DictConst.SALARYTYPE_DETAIL_WQZTKK);
            salaryMap.put(11,DictConst.SALARYTYPE_DETAIL_WQKGKK);
            salaryMap.put(12,DictConst.SALARYTYPE_DETAIL_WQSJKK);
            salaryMap.put(13,DictConst.SALARYTYPE_DETAIL_WQBJKK);
            salaryMap.put(14,DictConst.SALARYTYPE_DETAIL_WQLKKK);
            salaryMap.put(15,DictConst.SALARYTYPE_DETAIL_CDKK);
            salaryMap.put(16,DictConst.SALARYTYPE_DETAIL_ZTKK);
            salaryMap.put(17,DictConst.SALARYTYPE_DETAIL_KGKK);
            salaryMap.put(18,DictConst.SALARYTYPE_DETAIL_SJKK);
            salaryMap.put(19,DictConst.SALARYTYPE_DETAIL_BJKK);
            salaryMap.put(20,DictConst.SALARYTYPE_DETAIL_LKKK);
            salaryMap.put(21,DictConst.SALARYTYPE_DETAIL_GZRJBBT);
            salaryMap.put(22,DictConst.SALARYTYPE_DETAIL_XXRJBBT);
            salaryMap.put(23,DictConst.SALARYTYPE_DETAIL_JJRJBBT);
            salaryMap.put(24,DictConst.SALARYTYPE_DETAIL_DKYLJ);
            salaryMap.put(25,DictConst.SALARYTYPE_DETAIL_DKSYJ);
            salaryMap.put(26,DictConst.SALARYTYPE_DETAIL_DKYB);
            salaryMap.put(27,DictConst.SALARYTYPE_DETAIL_DKGJJ);
            salaryMap.put(28,DictConst.SALARYTYPE_DETAIL_GRSDS);

            /* 插入数据库表中 */
            saveReportInfo(comId, reportMonth,xlsList,validPerIdList,perKeyMap,salaryMap,paramDTO);

        } catch (IOException e) {
            e.printStackTrace();
            return Result.fail(PublicConst.OP_IMPORT_FAILURE);
        }

        return Result.success(PublicConst.OP_SUCCESS);
    }

    /**
     * 校验excel报表数据
     *
     * @return
     */
    private List<String> validateReportInfo(List<List<Object>> xlsList,Map<Integer,String> requiredMap
            ,Map<Integer,String> digitMap,List<String> validPerIdList,List<String> perKeyList,Map<String,String> perKeyMap) {
        // 错误信息集合返回
        List<String> errorList = Lists.newArrayList();
        /* 循环excel的数据 */
        for (int i = 3; i < xlsList.size(); i++) {
            /* 得到每一行数据 */
            List<Object> row = xlsList.get(i);

            String perName = "";
            String perKey = "";
            /* 循环每一行的数据 */
            int colIndex = 0;
            for (Object col:row) {
                /* 该项未录入任何值 */
                if (col == null ) {
                    if (requiredMap.containsKey(colIndex)) {
                        errorList.add("第" + (i + 1) + "行，【"+requiredMap.get(colIndex)+"】列数据不能为空，请检查");
                    }
                    colIndex++;
                    continue;
                }
                String value = col.toString().trim();

                /* 校验字段是否为空 */
                if (StrUtil.isEmpty(value) && requiredMap.containsKey(colIndex)) {
                    errorList.add("第" + (i + 1) + "行，【"+requiredMap.get(colIndex)+"】列数据不能为空，请检查");
                    colIndex++;
                    continue;
                }

                if (StrUtil.isEmpty(col.toString())) {
                    colIndex++;
                    continue;
                }

                /* 需要校验数字类型 */
                if (digitMap.containsKey(colIndex)) {
                    if (!NumberUtil.isNumber(value)) {
                        errorList.add("第" + (i + 1) + "行，【"+digitMap.get(colIndex)+"】列应为数字，请检查");
                        colIndex++;
                        continue;
                    }
                }

                /* 姓名 */
                if ( colIndex == 0 ) {
                    perKey = value;
                    perName = value;
                }
                /* 证件号 */
                else if (colIndex == 1) {
                    perKey = perKey + value;
                }
                colIndex++;
            }
            /* 查询未匹配到人的记录 */
            if (!perKeyList.contains(perKey)) {
                errorList.add("第" + (i + 1) + "行，未匹配到公司员工【"+perName+"】，请检查");
            }
            /* 符合条件的perId */
            else {
                validPerIdList.add(perKeyMap.get(perKey));
            }
        }
        return errorList;
    }

    /**
     * 校验excel报表数据
     *
     * @return
     */
    private Result saveReportInfo(Long comId, String month, List<List<Object>> xlsList,
                                  List<String> validPerIdList,Map<String,String> perKeyMap,Map<Integer,String> salaryMap,
                                  ParamDTO dto) {
        List<PerSalaryGrant> grantList = Lists.newArrayList();
        List<PerSalaryGrantDetail> detailList = Lists.newArrayList();
        /* 从第4行开始，因为前三行是标题行 */
        for (int i = 3; i < xlsList.size(); i++) {
            /* 得到每一行数据 */
            List<Object> row = xlsList.get(i);
            long perSalaryGrantId = idWorker.nextId();
            PerSalaryGrant grant = new PerSalaryGrant();
            grant.setPerSalaryGrantId(perSalaryGrantId);
            grant.setComId(comId);
            grant.setGrantYears(month);
            grant.setPerId(Long.parseLong(perKeyMap.get(String.valueOf(row.get(0)) + String.valueOf(row.get(1)))));
            grant.setIsConfirm(CommonConst.N);
            grant.setActualAmount(BigDecimal.ZERO);
            if (row.size() > 31) {
                grant.setPayableAmount(safeTransformBigDecimal(row.get(31)));
            }
            if (row.size() > 29) {
                grant.setSumAmount(safeTransformBigDecimal(row.get(29)));
            }
            if (row.size() > 30) {
                grant.setTaxableAmount(safeTransformBigDecimal(row.get(30)));
            }
            /* 如果前面未对几个关键的薪酬项赋值，则默认赋值为0 */
            if (grant.getPayableAmount() == null) {
                grant.setPayableAmount(BigDecimal.ZERO);
            }
            if (grant.getSumAmount() == null) {
                grant.setSumAmount(BigDecimal.ZERO);
            }
            if (grant.getTaxableAmount() == null) {
                grant.setTaxableAmount(BigDecimal.ZERO);
            }
            grant.setSalaryTempName("线下导入");
            grantList.add(grant);

            /* 开始解析每一条数据中的每一项数据 */
            Date now = new Date();
            /* 循环薪酬项 */
            for (int j = 2; j <= row.size(); j++) {
                /* 过滤不是薪酬项的内容 */
                if (!salaryMap.containsKey(j)) {
                    continue;
                }
                PerSalaryGrantDetail detail = new PerSalaryGrantDetail();
                detail.setPerSalaryGrantId(perSalaryGrantId);
                detail.setSalaryType(salaryMap.get(j));
                if (row.size() > j) {
                    detail.setAmount(safeTransformBigDecimal(row.get(j)));
                }
                /* 如果前面未对薪酬项赋值，则默认赋值为0 */
                if (detail.getAmount() == null) {
                    detail.setAmount(BigDecimal.ZERO);
                }
                detail.setCreateTime(now);
                detailList.add(detail);
            }
        }

        /* 先删除原有的数据 */
        List<Long> grantIdList = perSalaryGrantMapper.getGrantByPerIdList(comId, month, validPerIdList);
        if ( CollectionUtil.isNotEmpty(grantIdList) ) {
            perSalaryGrantDetailMapper.deleteGrantDetailByPerIdList(comId, month, grantIdList);
        }
        if (CollectionUtil.isNotEmpty(validPerIdList)) {
            perSalaryGrantMapper.deletePerGrant(comId,month,validPerIdList);
        }

        /* 保存 */
        perSalaryGrantMapper.batchInsertPerSalaryGrantByPerIdList(grantList);
        perSalaryGrantDetailMapper.batchInsertPerSalaryGrantDetailByList(detailList);

        ComSalaryRecord comSalaryRecord = new ComSalaryRecord();
        comSalaryRecord.setComSalaryRecordId(idWorker.nextId());
        comSalaryRecord.setCreateTime(new Date());
        comSalaryRecord.setCreateBy(dto.getCreateBy());
        comSalaryRecord.setGrantYears(month);
        comSalaryRecord.setIsCreateData(CommonConst.N);
        comSalaryRecord.setComId(comId);
        comSalaryRecordMapper.insertComSalaryRecord(comSalaryRecord);

        return Result.success(CommonConst.OP_SUCCESS);
    }


    /**
     * 生成失败的薪酬导入信息excel
     */
    private String buildFailExcel(List<String> errorList) throws BusinessException {
        try {
            String uploadPath = cacheService.getSystemConfig("SYS_PIC_PATH");
            String os = System.getProperty("os.name");
            if (os.toLowerCase().startsWith("win")) {
                uploadPath = "C:" + uploadPath;
            }

            String dirPath = uploadPath + "temp/" + DateUtil.today();
            FileUtil.mkdir(dirPath);

            String fileName = DateUtil.current() + ".xls";
            String filePath = dirPath + "/" + fileName;

            // 通过工具类创建writer ，jdk7 默认关闭writer对象
            try ( ExcelWriter writer = cn.hutool.poi.excel.ExcelUtil.getWriter(filePath) ) {
                /* 设置列宽 */
                writer.setColumnWidth(0,80);
                // 一次性写出内容，使用默认样式
                writer.write(errorList);
            }
            File file = new File(filePath);
            String platform = cacheService.getSystemConfig("PLATFORM_SIGN");
            String aliUrl = BcxinFileUtils.huaweiOBSFileUpload(file,platform,"xls");
            file.delete();
            return aliUrl;

        } catch (Exception e) {
            e.printStackTrace();
            throw new BusinessException(Result.ERROR, "生成错误失败人员文件错误");
        }
    }

    /**
     * 安全转换 obj to BigDecimal
     * */
    private BigDecimal safeTransformBigDecimal(Object value){
        if (value == null) {
            return BigDecimal.ZERO;
        }
        if (StringUtil.isEmpty(value.toString()) || Objects.equals(value.toString(),CommonConst.N)) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(value.toString());
    }
}