package com.zbkj.front.controller;

import com.zbkj.common.request.OrderPayRequest;
import com.zbkj.common.request.SendCodeRequest;
import com.zbkj.common.response.OrderPayResultResponse;
import com.zbkj.common.response.PayConfigResponse;
import com.zbkj.common.result.CommonResult;
import com.zbkj.service.service.PayService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.concurrent.TimeUnit;

/**
 * 支付控制器
 * +----------------------------------------------------------------------
 * | CRMEB [ CRMEB赋能开发者，助力企业发展 ]
 * +----------------------------------------------------------------------
 * | Copyright (c) 2016~2023 https://www.crmeb.com All rights reserved.
 * +----------------------------------------------------------------------
 * | Licensed CRMEB并不是自由软件，未经许可不能去掉CRMEB相关版权
 * +----------------------------------------------------------------------
 * | Author: CRMEB Team <admin@crmeb.com>
 * +----------------------------------------------------------------------
 */
@Slf4j
@RestController
@RequestMapping("api/front/pay")
@Api(tags = "支付管理")
public class PayController {

    @Autowired
    private PayService payService;
    @Autowired
    private RedissonClient redisson;

    @ApiOperation(value = "获取支付配置")
    @RequestMapping(value = "/get/config", method = RequestMethod.GET)
    public CommonResult<PayConfigResponse> getPayConfig() {
        return CommonResult.success(payService.getPayConfig());
    }

    @ApiOperation(value = "订单支付")
    @RequestMapping(value = "/payment", method = RequestMethod.POST)
    public CommonResult<OrderPayResultResponse> payment(@RequestBody @Validated OrderPayRequest orderPayRequest) {
        String lockId = "TRANSPAYMENT"+orderPayRequest.getOrderNo();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                return CommonResult.success(payService.payment(orderPayRequest));
            }else{
                return CommonResult.failed("订单支付中，请勿重复操作");
            }
        } catch (Exception e) {
            log.error("payment error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    @ApiOperation(value = "查询订单微信支付结果")
    @RequestMapping(value = "/query/wechat/pay/result/{orderNo}", method = RequestMethod.GET)
    public CommonResult<Boolean> queryPayResult(@PathVariable(value = "orderNo") String orderNo) {
        return CommonResult.success(payService.queryWechatPayResult(orderNo));
    }

    @ApiOperation(value = "查询订单支付宝支付结果")
    @RequestMapping(value = "/query/ali/pay/result/{orderNo}", method = RequestMethod.GET)
    public CommonResult<Boolean> queryAliPayResult(@PathVariable(value = "orderNo") String orderNo) {
        return CommonResult.success(payService.queryAliPayResult(orderNo));
    }

    @ApiOperation(value = "发送短信钱包验证码")
    @RequestMapping(value = "/send/code", method = RequestMethod.POST)
    public CommonResult<String> sendCode(@RequestBody @Validated SendCodeRequest request) {
        if (payService.sendWalletCode(request.getPhone())) {
            return CommonResult.success("发送成功");
        }
        return CommonResult.failed("发送失败");
    }

    @ApiOperation(value = "查询订单支付结果")
    @RequestMapping(value = "/query/pay/result/{orderNo}", method = RequestMethod.GET)
    public CommonResult<Boolean> checkOrderPaySuccess(@PathVariable(value = "orderNo") String orderNo){
        return CommonResult.success(payService.checkPayResult(orderNo));
    }
}
