package com.bcxin.obpm.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.HexUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.auth.common.constant.Constants;
import com.bcxin.auth.common.core.domain.AjaxResult;
import com.bcxin.auth.common.exception.V5BusinessException;
import com.bcxin.auth.common.utils.DateUtils;
import com.bcxin.auth.common.utils.HttpUtil;
import com.bcxin.auth.common.utils.StringUtils;
import com.bcxin.auth.system.domain.SysInterfaceLog;
import com.bcxin.auth.system.mapper.SysInterfaceLogMapper;
import com.bcxin.obpm.dto.*;
import com.bcxin.obpm.service.BackgroundService;
import com.bcxin.obpm.util.AuthConstants;
import com.bcxin.obpm.util.FaceUtil;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;


@Service("backgroundService_140500")
public class JcBackgroundServiceImpl implements BackgroundService {
    private static Logger logger = LoggerFactory.getLogger(JcBackgroundServiceImpl.class);

    @Autowired
    private FaceUtil faceUtil;

    @Autowired
    private SysInterfaceLogMapper sysInterfaceLogMapper;

    /**
     * 该类实现了人员审查的背景服务。它检查一个人是否有任何背景信息，如吸毒、在逃或有犯罪记录。
     * 检查是使用JC背景审查API完成的。
     */
    @Override
    public AjaxResult personnelReview(String idNumber, String realName) {
        if (StringUtils.isEmpty(idNumber)) {
            return AjaxResult.error("参数不正确");
        }
        List<BackgroundInfo> backgroundInfoList = new ArrayList<>();
        BackgroundResult backgroundResult = new BackgroundResult();
        checkPersonInfo(backgroundInfoList, idNumber);
        backgroundResult.setIdNumber(idNumber);
        backgroundResult.setCensorResult(backgroundInfoList);
        logger.info("身份证{}筛查结果:{}", idNumber, backgroundResult);
        // 请求
        return AjaxResult.success("请求成功", backgroundResult);
    }

    /**
     * 此方法检查人员的背景信息，如吸毒、在逃或有犯罪记录。
     * 详细信息是使用JC背景审查API获取的，并添加到列表中。
     *
     * @param list     存储背景信息的列表。
     * @param idNumber 人员的身份证号码。
     */
    private void checkPersonInfo(List<BackgroundInfo> list, String idNumber) {
        BackgroundInfo backgroundInfo = new BackgroundInfo();
        String wfStr = StringUtils.EMPTY;
        try {
            Map<String, Object> wfMap = new HashMap<>();

            // 检查是否重点人
            boolean isPointsUser = getJcCensorResult(idNumber, Constants.JC_PEOPLE_BACKGROUND_ZDR_COMPARE_URL);
            if (isPointsUser) {
                wfStr += "重点人员,";
            }

            // 检查是否在逃
            boolean isFugitive = getJcCensorResult(idNumber, Constants.JC_PEOPLE_BACKGROUND_ZTRY_COMPARE_URL);
            if (isFugitive) {
                wfStr += "在逃人员,";
            }

            // 检查是否有违法犯罪记录
            boolean hasCriminalRecord = getJcCensorResult(idNumber, Constants.JC_PEOPLE_BACKGROUND_CASE_COMPARE_URL);
            if (hasCriminalRecord) {
                wfStr += "有违法犯罪记录,";
            }

            if (!wfStr.isEmpty()) {
                wfStr = wfStr.substring(0, wfStr.length() - 1);
                wfMap.put("detail", wfStr);
                backgroundInfo.setCode(200);
                backgroundInfo.setData(wfMap);
                backgroundInfo.setResult(1);
                backgroundInfo.setCxxt("晋城市公安局服务总线接口");
                list.add(backgroundInfo);
            }
        } catch (Exception e) {
            throw new V5BusinessException(e.getMessage());
        }
    }

    /**
     * 此方法查询JC背景审查API，根据给定的类型检查人员是否有任何背景信息。
     *
     * @param idNumber 人员的身份证号码。
     * @param requestUrl 晋城市公安局服务总线接口地址。
     * @return 如果人员具有指定的背景信息，则返回true；否则返回false。
     */
    private boolean getJcCensorResult(String idNumber, String requestUrl) {
        // 创建接口日志对象
        SysInterfaceLog interfaceLog = new SysInterfaceLog();
        interfaceLog.setCreateTime(new Date());
        interfaceLog.setActive(true);
        interfaceLog.setUrl(requestUrl);
        interfaceLog.setParameters("晋城市公安局服务总线接口：" + idNumber);

        // 创建请求参数对象
        requestUrl = String.format(requestUrl, idNumber);
        // 创建头部信息
        Map<String, String> headerMap = this.getHeaderMap();
        logger.error("requestUrl:{}", requestUrl);
        logger.error("headerMap:{}", JSONObject.toJSONString(headerMap));

        for (int i = 0; i < Constants.ZAFLAG_COUNT; i++) {
            String result = HttpUtil.get(requestUrl, headerMap);
            logger.error("晋城市公安局服务总线接口，背景筛查，身份证：{}，返回结果：{}", idNumber, result);

            JcCensorResult jcCensorResult = JSONObject.parseObject(result, JcCensorResult.class);
            if (jcCensorResult != null && jcCensorResult.getCode() == 200) {
                // 记录接口日志
                interfaceLog.setResult(result);
                interfaceLog.setUpdateTime(new Date());
                interfaceLog.setSuccessful(true);
                // 异步插入接口日志
                new Thread(() -> {
                    sysInterfaceLogMapper.insert(interfaceLog);
                }).start();
                return CollectionUtil.isNotEmpty(jcCensorResult.getData());
            }
        }
        // 记录接口日志
        interfaceLog.setResult("背景筛查新库查询异常!!");
        interfaceLog.setUpdateTime(new Date());
        interfaceLog.setSuccessful(false);
        // 异步插入接口日志
        new Thread(() -> {
            sysInterfaceLogMapper.insert(interfaceLog);
        }).start();
        throw new V5BusinessException("背景筛查新库查询异常！！");
    }

    @Override
    public AjaxResult getPersonnelInformation(String idNumber) {
        if (StringUtils.isEmpty(idNumber)) {
            return AjaxResult.error("参数不正确");
        }
        AuthenticationResult authenticationResult = new AuthenticationResult();
        authenticationResult.setIdNumber(idNumber);
        try {
            PerPhotoDto perPhotoDto = getPopulationInfo(idNumber, null);
            authenticationResult.setPhoto(perPhotoDto.getPhoto());
            authenticationResult.setRealName(perPhotoDto.getName());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new V5BusinessException("晋城市公安局服务总线接口调用异常！！");
        }
        logger.info("{}人员信息服务返回:{}", idNumber, authenticationResult);
        return AjaxResult.success("请求成功", authenticationResult);
    }

    /**
     * 获取人口库信息
     *
     * @param idnum
     * @return
     */
    private PerPhotoDto getPopulationInfo(String idnum, String name) {
        PerPhotoDto person = new PerPhotoDto();
        JcPerInfoDto jcPerInfoDto = getJcPerInfo(idnum, name);
        if (jcPerInfoDto != null) {
            person.setPhoto(jcPerInfoDto.getIdCardUrl());
            person.setName(jcPerInfoDto.getName());
            person.setIdNum(jcPerInfoDto.getIdNo());
        }
        return person;
    }

    /**
     * 根据身份证号码获取人口信息
     */
    private JcPerInfoDto getJcPerInfo(String idNumber, String name) {
        // 创建接口日志对象
        SysInterfaceLog interfaceLog = new SysInterfaceLog();
        interfaceLog.setCreateTime(new Date());
        interfaceLog.setActive(true);
        interfaceLog.setUrl(Constants.JC_PEOPLE_IDENTITY_VERIFY_URL);
        interfaceLog.setParameters("晋城市公安局服务总线接口：" + idNumber);

        // 创建请求参数对象
        String requestUrl = String.format(Constants.JC_PEOPLE_IDENTITY_VERIFY_URL, idNumber, name);
        // 创建头部信息
        Map<String, String> headerMap = this.getHeaderMap();
        logger.error("晋城实名认证url:{}", requestUrl);
        logger.error("晋城实名认证headerMap:{}", JSONObject.toJSONString(headerMap));

        // 初始化错误计数
        // 发起请求，最多尝试 Constants.ZAFLAG_COUNT 次
        for (int i = 0; i < Constants.ZAFLAG_COUNT; i++) {
            // 发起请求
            String result = HttpUtil.get(requestUrl, headerMap);
            logger.error("晋城市公安局服务总线接口，实名认证，身份证：{}，返回结果：{}", idNumber, result);
            // 解析返回结果
            JcCensorResult jcCensorResult = JSONObject.parseObject(result, JcCensorResult.class);
            // 判断返回结果是否成功
            if (jcCensorResult != null && jcCensorResult.getCode() == 200) {
                // 获取人员列表
                List<JSONObject> data = jcCensorResult.getData();
                if (data.size() > 0) {
                    // 获取第一个人员信息
                    JcPerInfoDto person = data.get(0).toJavaObject(JcPerInfoDto.class);
                    // 记录接口日志
                    interfaceLog.setResult(result);
                    interfaceLog.setUpdateTime(new Date());
                    interfaceLog.setSuccessful(true);
                    // 异步插入接口日志
                    new Thread(() -> {
                        sysInterfaceLogMapper.insert(interfaceLog);
                    }).start();

                    return person;
                }
            }
        }
        // 如果错误计数达到3次，则记录接口日志
        interfaceLog.setResult("获取人口库基本信息异常!!");
        interfaceLog.setUpdateTime(new Date());
        interfaceLog.setSuccessful(false);
        new Thread(() -> {
            sysInterfaceLogMapper.insert(interfaceLog);
        }).start();
        return null;
    }

    @Override
    public List<AuthLog> validateResult(List<AuthLog> list) {
        for (AuthLog authLog : list) {
            try {
                this.matchResult(authLog);
                authLog.setAuthDate(DateUtils.getNowDate());
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
            }
        }
        return list.stream().filter(((authLog) -> StrUtil.isNotEmpty(authLog.getAuthStatus()))).collect(Collectors.toList());
    }

    @Override
    public void getPersonnelAddress(String idNumber) {
    }

    /**
     * 实名认证比对
     *
     * @param authLog
     * @throws IOException
     */
    public void matchResult(AuthLog authLog) throws IOException {
        try {
            /***2.2 获取人口库照片，base64，如果是互联网则获取本地图片，以身份证命名******/
            PerPhotoDto person = getPopulationInfo(authLog.getIdNumber(), authLog.getSecurityName());
            String xp = "";
            /**
             * 2.2.1获取姓名
             */
            String name = person.getName();
            /**
             * 2.2.2 判断姓名是否为空，如果为空，则人口库没有信息
             */
            if (StrUtil.isEmpty(name)) {
                // 姓名和身份证不匹配
                authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_CARD_IDEMSG);
            } else if (name.equals(authLog.getSecurityName())) {
                /**
                 * 2.2.3 匹配姓名是否正常，如果匹配则获取照片信息，不匹配则返回
                 */
                // 获取人口库照片
                xp = person.getPhoto();
                /***2.2.4 没有照片代码身份证和姓名不匹配**/
                if (StrUtil.isEmpty(xp)) {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_PHOTO_IDEMSG);
                }
            } else {
                /***
                 * 2.2.4 姓名和身份证不匹配
                 */
                authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_NAME_IDEMSG);
            }
            /***
             * 2.3 如果有照片进行对比
             */
            if (StrUtil.isNotEmpty(xp)) {
                /***2.4 人口库和认证头像比对 **/
                if (faceUtil.matchResult(xp, faceUtil.ImageToBase64(authLog.getImgPath()))) {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_YES);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_YES_MESSAGE);
                } else {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_IMGMSG);
                }
            }
            authLog.setAuthDate(DateUtils.getNowDate());
        } catch (Exception e) {
            authLog.setAuthStatus(AuthConstants.AUTHRESULT_ZHONG);
            authLog.setAuthResult("比对接口异常！！");
            e.printStackTrace();
        }
    }



    /**
     * description：获取晋城市公安局服务总线接口请求头部信息
     * author：linchunpeng
     * date：2025/5/7
     */
    private Map<String, String> getHeaderMap() {
        Map<String, String> headerMap = Maps.newHashMap();
        //访问密钥
        String ak = "QkzBz5Z5BxCFjXNs";
        headerMap.put("X-Ca-AccessKey", ak);
        //安全密钥
        String sk = "58EpDtsKwWNNinxwbwwpFXw366SkcdpX";
        //请求编号
        String nonce = cn.hutool.core.lang.UUID.randomUUID().toString().replace("-", "");
        headerMap.put("X-Ca-Nonce", nonce);
        //请求时间戳
        String timestamp = System.currentTimeMillis() + "";
        headerMap.put("X-Ca-Timestamp", timestamp);
        String signStr = ak + nonce + timestamp + sk;
        //⽣成签名
        try {
            String sign = HexUtil.encodeHexStr(MessageDigest.getInstance("MD5").digest(signStr.getBytes(StandardCharsets.UTF_8)));
            headerMap.put("X-Ca-Sign", sign);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return headerMap;
    }
}