package com.wlos.app.utils;


import org.apache.pdfbox.cos.COSName;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.PDResources;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.PDXObject;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.apache.poi.hslf.usermodel.HSLFSlide;
import org.apache.poi.hslf.usermodel.HSLFSlideShow;
import org.apache.poi.hslf.usermodel.HSLFSlideShowImpl;
import org.apache.poi.xslf.usermodel.XMLSlideShow;
import org.apache.poi.xslf.usermodel.XSLFSlide;
import org.docx4j.Docx4J;
import org.docx4j.fonts.BestMatchingMapper;
import org.docx4j.fonts.Mapper;
import org.docx4j.fonts.PhysicalFonts;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.wlos.app.exception.BusinessException;
import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

public class PdfUtil {

    private static final Logger log = LoggerFactory.getLogger(PdfUtil.class);

    private static BufferedImage create(BufferedImage image, double angle,
                                        GraphicsConfiguration gc) {
        double sin = Math.abs(Math.sin(angle)), cos = Math.abs(Math.cos(angle));
        int w = image.getWidth(), h = image.getHeight();
        int neww = (int) Math.floor(w * cos + h * sin), newh = (int) Math.floor(h
                * cos + w * sin);
        int transparency = image.getColorModel().getTransparency();
        BufferedImage result = gc.createCompatibleImage(neww, newh, transparency);
        Graphics2D g = result.createGraphics();
        g.translate((neww - w) / 2, (newh - h) / 2);
        g.rotate(angle, w / 2, h / 2);
        g.drawRenderedImage(image, null);
        return result;
    }

    public static void pptxToPdf(File inputFile, File outputFile) throws Exception {
        //creating an empty presentation
        try (PDDocument document = new PDDocument();
             XMLSlideShow ppt = new XMLSlideShow(new FileInputStream(inputFile))) {
            //getting the dimensions and size of the slide
            Dimension pgsize = ppt.getPageSize();
            PDRectangle pdRectangle = new PDRectangle(pgsize.height, pgsize.width);
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            GraphicsDevice gd = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gd.getDefaultConfiguration();

            XSLFSlide[] slide = ppt.getSlides().toArray(new XSLFSlide[0]);

            for (int i = 0; i < slide.length; i++) {
                BufferedImage img = new BufferedImage(pgsize.width, pgsize.height, BufferedImage.TYPE_INT_RGB);
                Graphics2D graphics = img.createGraphics();

                //clear the drawing area
                graphics.setPaint(Color.white);
                graphics.fill(new Rectangle2D.Float(0, 0, pgsize.width, pgsize.height));

                //render
                slide[i].draw(graphics);

                BufferedImage img1 = create(img, -Math.PI / 2, gc);

                //Retrieving the page
                PDPage page = new PDPage(pdRectangle);
                page.setRotation(90);

                //Creating PDImageXObject object
                ByteArrayOutputStream imageByte = new ByteArrayOutputStream();
                ImageIO.write(img1,"png", imageByte);
                PDImageXObject pdImage = PDImageXObject.createFromByteArray(document, imageByte.toByteArray(), i + "");

                //creating the PDPageContentStream object
                PDPageContentStream contents = new PDPageContentStream(document, page);

                //Drawing the image in the PDF document
                contents.drawImage(pdImage, 0, 0, pgsize.height, pgsize.width);

                //Closing the PDPageContentStream object
                contents.close();

                document.addPage(page);
            }

            document.save(outputFile);
        }
    }

    public static void imageToPdf(File inputFile, File outputFile) throws Exception {
        //creating an empty presentation
        try (PDDocument document = new PDDocument()) {
            //getting the dimensions and size of the slide

            BufferedImage image = ImageIO.read(inputFile);
            PDRectangle pdRectangle = new PDRectangle(image.getWidth(), image.getHeight());

            //Retrieving the page
            PDPage page = new PDPage(pdRectangle);

            //Creating PDImageXObject object
            ByteArrayOutputStream imageByte = new ByteArrayOutputStream();
            ImageIO.write(image,"png", imageByte);
            PDImageXObject pdImage = PDImageXObject.createFromByteArray(document, imageByte.toByteArray(), inputFile.getName());

            //creating the PDPageContentStream object
            PDPageContentStream contents = new PDPageContentStream(document, page);

            //Drawing the image in the PDF document
            contents.drawImage(pdImage, 0, 0, image.getWidth(), image.getHeight());

            //Closing the PDPageContentStream object
            contents.close();

            document.addPage(page);

            document.save(outputFile);
        }
    }

    public static void pptToPdf(File inputFile, File outputFile) throws Exception {
        //creating an empty presentation
        try (PDDocument document = new PDDocument();
             HSLFSlideShow ppt = new HSLFSlideShow(new HSLFSlideShowImpl(new FileInputStream(inputFile)))) {
            //getting the dimensions and size of the slide
            Dimension pgsize = ppt.getPageSize();
            PDRectangle pdRectangle = new PDRectangle(pgsize.height, pgsize.width);
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            GraphicsDevice gd = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gd.getDefaultConfiguration();

            HSLFSlide[] slide = ppt.getSlides().toArray(new HSLFSlide[0]);

            for (int i = 0; i < slide.length; i++) {
                BufferedImage img = new BufferedImage(pgsize.width, pgsize.height, BufferedImage.TYPE_INT_RGB);
                Graphics2D graphics = img.createGraphics();

                //clear the drawing area
                graphics.setPaint(Color.white);
                graphics.fill(new Rectangle2D.Float(0, 0, pgsize.width, pgsize.height));

                //render
                slide[i].draw(graphics);

                BufferedImage img1 = create(img, -Math.PI / 2, gc);

                //Retrieving the page
                PDPage page = new PDPage(pdRectangle);
                page.setRotation(90);

                //Creating PDImageXObject object
                ByteArrayOutputStream imageByte = new ByteArrayOutputStream();
                ImageIO.write(img1,"png", imageByte);
                PDImageXObject pdImage = PDImageXObject.createFromByteArray(document, imageByte.toByteArray(), i + "");

                //creating the PDPageContentStream object
                PDPageContentStream contents = new PDPageContentStream(document, page);

                //Drawing the image in the PDF document
                contents.drawImage(pdImage, 0, 0, pgsize.height, pgsize.width);

                //Closing the PDPageContentStream object
                contents.close();

                document.addPage(page);
            }

            document.save(outputFile);
        }
    }

    public static void docxToPdf(File inputFile, File outputFile) throws Exception {
        WordprocessingMLPackage wordMLPackage = WordprocessingMLPackage.load(inputFile);

        Mapper fontMapper = new BestMatchingMapper();
        PhysicalFonts.getPhysicalFonts().forEach((k, v )-> log.info("PhysicalFonts {} - {} ", k, v.getName()));

        fontMapper.put("华文行楷", PhysicalFonts.get("STXingkai"));
        fontMapper.put("隶书", PhysicalFonts.get("LiSu"));
        fontMapper.put("宋体", PhysicalFonts.get("SimSun"));
        fontMapper.put("微软雅黑", PhysicalFonts.get("Microsoft Yahei"));
        fontMapper.put("黑体", PhysicalFonts.get("SimHei"));
        fontMapper.put("楷体", PhysicalFonts.get("KaiTi"));
        fontMapper.put("新宋体", PhysicalFonts.get("NSimSun"));
        fontMapper.put("华文行楷", PhysicalFonts.get("STXingkai"));
        fontMapper.put("华文仿宋", PhysicalFonts.get("STFangsong"));
        fontMapper.put("宋体扩展", PhysicalFonts.get("simsun-extB"));
        fontMapper.put("仿宋", PhysicalFonts.get("FangSong"));
        fontMapper.put("仿宋_GB2312", PhysicalFonts.get("FangSong_GB2312"));
        fontMapper.put("幼圆", PhysicalFonts.get("YouYuan"));
        fontMapper.put("华文宋体", PhysicalFonts.get("STSong"));
        fontMapper.put("华文中宋", PhysicalFonts.get("STZhongsong"));
        wordMLPackage.setFontMapper(fontMapper);
        Docx4J.toPDF(wordMLPackage, Files.newOutputStream(outputFile.toPath()));
    }



    public static void pdfExtractPictures(InputStream inputStream, int startIndex, int endIndex, String prefix, File outputFile) {

        try (PDDocument document = PDDocument.load(inputStream);
             ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(outputFile))) {
            int total = document.getPages().getCount();
            if (startIndex == -1) {
                startIndex = 0;
                endIndex = total - 1;
            }

            if (endIndex >= total) {
                endIndex = total - 1;
            }

            if (startIndex > endIndex) {
                throw new BusinessException("参数有误: 开始页面不能大于结束页面!");
            }

            int num = 1;
            for (int i = startIndex; i <= endIndex; i++) {
                PDPage pdfpage = document.getPage(i);
                // get resource of pdf
                PDResources pdResources = pdfpage.getResources();
                for (COSName cosName : pdResources.getXObjectNames()) {
                    PDXObject o = pdResources.getXObject(cosName);
                    if (o instanceof PDImageXObject) {
                        //得到BufferedImage对象
                        BufferedImage image = ((PDImageXObject) o).getImage();

                        zos.putNextEntry(new ZipEntry(prefix + (num++) + ".jpg"));

                        ByteArrayOutputStream imageByte = new ByteArrayOutputStream();
                        ImageIO.write(image, "png", imageByte);
                        InputStream fis = new ByteArrayInputStream(imageByte.toByteArray());
                        byte[] buffer = new byte[1024];
                        int r;
                        while ((r = fis.read(buffer)) != -1) {
                            zos.write(buffer, 0, r);
                        }
                        fis.close();
                    }
                }

            }
            zos.flush();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
    * @Description: pdf 提取图片为base64
    * @Param: [inputStream, startIndex, endIndex, prefix, outputFile]
    * @return: void
    * @Date: 2022/7/29
    */
    public static java.util.List<Map<String,String>> pdfExtractPictures(InputStream inputStream, int startIndex, int endIndex, String prefix) {
        java.util.List<Map<String,String>> result = new ArrayList<>();
        try (PDDocument document = PDDocument.load(inputStream)) {
            //分页读取
            int total = document.getPages().getCount();
            if (startIndex == -1) {
                startIndex = 0;
                endIndex = total - 1;
            }

            if (endIndex >= total) {
                endIndex = total - 1;
            }

            if (startIndex > endIndex) {
                throw new RuntimeException("参数有误: 开始页面不能大于结束页面!");
            }

            int num = 1;
            for (int i = startIndex; i <= endIndex; i++) {
                PDPage pdfpage = document.getPage(i);
                // get resource of pdf
                PDResources pdResources = pdfpage.getResources();
                for (COSName cosName : pdResources.getXObjectNames()) {
                    PDXObject o = pdResources.getXObject(cosName);
                    if (o instanceof PDImageXObject) {
                        //得到BufferedImage对象
                        BufferedImage image = ((PDImageXObject) o).getImage();
                        String fileName = prefix + (num++) + ".jpg";

                        ByteArrayOutputStream imageByte = new ByteArrayOutputStream();
                        ImageIO.write(image, "png", imageByte);
                        //转base64
                        byte[] bytes = imageByte.toByteArray();
                        String encode = Base64.getEncoder().encodeToString(bytes);
                        Map<String, String> map = new HashMap<>();
                        map.put("fileName",fileName);
                        map.put("bytes",encode);
                        result.add(map);
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return result;
    }

}
