package com.wlos.app.utils;

import cn.hutool.core.date.DateUtil;
import com.wlos.app.exception.BusinessException;
import org.apache.commons.lang3.ObjectUtils;

import java.time.*;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalUnit;

public class BlocklyDateUtils {

    static LocalDateTime getLocalDateTime(Object date) {
        return DateUtil.parseDate(String.valueOf(date)).toLocalDateTime();
    }

    static LocalDate getLocalDate(Object date) {
        return DateUtil.parseDate(String.valueOf(date)).toLocalDateTime().toLocalDate();
    }

    static LocalTime getLocalTime(Object date) {
        return DateUtil.parseDate(String.valueOf(date)).toLocalDateTime().toLocalTime();
    }

    static TemporalUnit getTemporalUnit(String unit){
        return switch (unit) {
            case "year": yield ChronoUnit.YEARS;
            case "quarter":
            case "months":
                yield ChronoUnit.MONTHS;
            case "weeks": yield ChronoUnit.WEEKS;
            case "days": yield ChronoUnit.DAYS;
            case "hours": yield ChronoUnit.HOURS;
            case "minutes": yield ChronoUnit.MINUTES;
            case "seconds": yield ChronoUnit.SECONDS;
            default:
                throw new BusinessException("Unexpected value: " + unit);
        };
    }

    static int getAmountToAdd(Object add, String unit) {
        if (ObjectUtils.isEmpty(add)) {
            return 0;
        }
        try {
            int amountToAdd = Integer.parseInt(String.valueOf(add));
            if (unit.equals("quarter")) {
                amountToAdd = amountToAdd * 3;
            }
            return amountToAdd;
        } catch (Exception e) {
            return 0;
        }
    }

    /**
     * 日期加减
     * @param date
     * @param add
     * @param mode
     * @param unit
     * @return
     */
    public static String dateArithmetic(Object date, Object add, String mode, String unit) {
        if (ObjectUtils.isEmpty(date)) {
            return null;
        }
        LocalDate localDate = getLocalDate(date);
        if (mode.equals("add")) {
            return localDate.plus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        } else {
            return localDate.minus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        }
    }

    /**
     * 日期时间加减
     * @param date
     * @param add
     * @param mode
     * @param unit
     * @return
     */
    public static String dateTimeArithmetic(Object date, Object add, String mode, String unit) {
        if (ObjectUtils.isEmpty(date)) {
            return null;
        }
        LocalDateTime localDate = getLocalDateTime(date);
        if (mode.equals("add")) {
            return localDate.plus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        } else {
            return localDate.minus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        }
    }

    /**
     * 时间加减
     * @param date
     * @param add
     * @param mode
     * @param unit
     * @return
     */
    public static String timeArithmetic(Object date, Object add, String mode, String unit) {
        if (ObjectUtils.isEmpty(date)) {
            return null;
        }
        LocalTime localDate = getLocalTime(date);
        if (mode.equals("add")) {
            return localDate.plus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        } else {
            return localDate.minus(getAmountToAdd(add, unit), getTemporalUnit(unit)).toString();
        }
    }

    /**
     * 系统时间当月天数
     * @return
     */
    public static int getCurrentDateLengthOfMonth(){
        return LocalDate.now().lengthOfMonth();
    }

    /**
     * 获取指定单位的第一天的日期
     * @param mode
     * @return
     */
    public static String getCurrentDateFirstDay(String mode){
        return switch (mode) {
            case "year":
                yield LocalDate.now().with(TemporalAdjusters.firstDayOfYear()).toString();
            case "month":
                yield LocalDate.now().with(TemporalAdjusters.firstDayOfMonth()).toString();
            case "quarter":
                yield LocalDate.of(LocalDate.now().getYear(), LocalDate.now().getMonth().firstMonthOfQuarter().getValue(), 1).toString();
            case "week":
                yield LocalDate.now().with(DayOfWeek.MONDAY).toString();
            default:
                throw new IllegalStateException("Unexpected value: " + mode);
        };
    }

    /**
     * 获取指定单位的最后一天的日期
     * @param mode
     * @return
     */
    public static String getCurrentDateLastDay(String mode){
        return switch (mode) {
            case "year":
                yield LocalDate.now().with(TemporalAdjusters.lastDayOfYear()).toString();
            case "month":
                yield LocalDate.now().with(TemporalAdjusters.lastDayOfMonth()).toString();
            case "quarter":
                yield getEndDayOfQuarter();
            case "week":
                yield LocalDate.now().with(DayOfWeek.SUNDAY).toString();
            default:
                throw new IllegalStateException("Unexpected value: " + mode);
        };
    }

    /**
     * 获取本季度最后一天
     * @return
     */
    static String getEndDayOfQuarter(){
        LocalDate today=LocalDate.now();
        Month month = today.getMonth();
        Month firstMonthOfQuarter = month.firstMonthOfQuarter();
        Month endMonthOfQuarter = Month.of(firstMonthOfQuarter.getValue() + 2);
        return LocalDate.of(today.getYear(), endMonthOfQuarter, endMonthOfQuarter.length(today.isLeapYear())).toString();
    }

    /**
     * 获取指定时间的单位值：年/月/日/时/分/秒
     * @param date
     * @param unit
     * @return
     */
    public static int getDateTimeFieldByUnit(Object date, String unit) {
        if (ObjectUtils.isEmpty(date)) {
            return 0;
        }
        LocalDateTime localDate = getLocalDateTime(date);
        return switch (unit) {
            case "year":
                yield localDate.getYear();
            case "month":
                yield localDate.getMonthValue();
            case "day":
                yield localDate.getDayOfMonth();
            case "hour":
                yield localDate.getHour();
            case "minute":
                yield localDate.getMinute();
            case "second":
                yield localDate.getSecond();
            default:
                throw new IllegalStateException("Unexpected value: " + unit);
        };
    }

    /**
     * 日期转换
     * @param date
     * @param mode
     * @return
     */
    public static String transferDate(Object date, String mode) {
        if (ObjectUtils.isEmpty(date)) {
            return null;
        }
        LocalDateTime localDate = getLocalDateTime(date);
        return switch (mode) {
            case "time":
                yield localDate.toLocalTime().toString();
            case "date":
                yield localDate.toLocalDate().toString();
            case "datetime":
            case "text":
                yield localDate.toString();
            default:
                throw new IllegalStateException("Unexpected value: " + mode);
        };
    }

    /**
     * 获取2个时间之间的差值
     *
     * @param date1
     * @param date2
     * @param unit
     * @return
     */
    public static long getDatePeriod(Object date1, Object date2, String unit) {
        if (ObjectUtils.isEmpty(date1) || ObjectUtils.isEmpty(date2)) {
            return 0;
        }
        LocalDateTime localDate1 = getLocalDateTime(date1);
        LocalDateTime localDate2 = getLocalDateTime(date2);
        //判断时间大小
        /*if (localDate2.isBefore(localDate1)) {
            LocalDateTime ldt = localDate2;
            localDate2 = localDate1;
            localDate1 = ldt;
        }*/
        return switch (unit) {
            case "year":
                yield localDate1.until(localDate2, ChronoUnit.YEARS);
            case "months":
                yield localDate1.until(localDate2, ChronoUnit.MONTHS);
            case "day":
                yield localDate1.until(localDate2, ChronoUnit.DAYS);
            case "hour":
                yield localDate1.until(localDate2, ChronoUnit.HOURS);
            case "minute":
                yield localDate1.until(localDate2, ChronoUnit.MINUTES);
            case "second":
                yield localDate1.until(localDate2, ChronoUnit.SECONDS);
            case "millis":
                yield localDate1.until(localDate2, ChronoUnit.MILLIS);
            default:
                throw new IllegalStateException("Unexpected value: " + unit);
        };
    }

}
