package com.wlos.app.bl.impl;

import com.wlos.app.bl.ParseService;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;


@Service
public class CsvParseService implements ParseService {

    private final Logger logger = LoggerFactory.getLogger(CsvParseService.class);
    private final Map<String, Integer> headerIndex = new HashMap<>();
    private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
    private final ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();
    private final List<String> excludeFields = List.of("serialVersionUID", "id");

    private void initializeHeader(String[] headers) {
        for (int i = 0; i < headers.length; i++) {
            headerIndex.put(headers[i], i);
        }
    }

    @Override
    public boolean support(String fileType) {
        return "csv".equals(fileType) || "txt".equals(fileType);
    }

    @Override
    public List<Object> parse(Class<Object> clazz, byte[] file) throws Exception {
        try (BufferedReader csvReader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(file)))) {
            String headerRow = csvReader.readLine();
            if (headerRow != null) {
                initializeHeader(headerRow.split(","));
            }

            List<Object> list = new ArrayList<>();
            String row;
            while ((row = csvReader.readLine()) != null) {
                Object o = clazz.getDeclaredConstructor().newInstance();
                String[] data = row.split(",");
                headerIndex.forEach((k, v) ->
                        {
                            try {
                                Field field = clazz.getDeclaredField(k);
                                field.setAccessible(true);
                                if (field.getType() == Date.class) {
                                    if (StringUtils.isNotBlank(data[v])) {
                                        field.set(o, simpleDateFormat.parse(data[v]));
                                    }
                                } else {
                                    field.set(o, convertUtilsBean.convert(data[v], field.getType()));
                                }

                            } catch (NoSuchFieldException | IllegalAccessException | ParseException e) {
                                e.printStackTrace();
                            }
                        }
                );
                list.add(o);
            }
            return list;
        }
    }

    @Override
    public byte[] importTemplate(Class<Object> clazz) {
        List<String> fields = Arrays.stream(clazz.getDeclaredFields())
                .map(Field::getName)
                .filter(it -> !excludeFields.contains(it))
                .collect(Collectors.toList());
        return String.join(",", fields).getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public byte[] exportModel(Class<Object> clazz, List<Object> list, List<String> headers, UnaryOperator<Object> fun) {
        StringBuilder stringBuilder = new StringBuilder();
        List<Method> methods = Arrays.stream(clazz.getMethods())
                .filter(it -> it.getName().startsWith("get") && !it.getName().equals("getClass"))
                .filter(it -> {
                    String header = (char) (it.getName().charAt(3) + 32) + it.getName().substring(4);
                    if (headers.contains(header)) {
                        stringBuilder.append(header).append(",");
                        return true;
                    }
                    return false;
                })
                .collect(Collectors.toList());

        stringBuilder.deleteCharAt(stringBuilder.length() - 1);

        list.forEach(it ->
                {
                    stringBuilder.append("\n");
                    methods.forEach(method -> {
                        try {
                            Object v = method.invoke(it);
                            logger.debug("csv export method {} {}", method.getName(), v);
                            if (method.getReturnType() == Date.class) {
                                if (v != null) {
                                    stringBuilder.append(simpleDateFormat.format(v));
                                }
                            } else {
                                stringBuilder.append(convertUtilsBean.convert(v, String.class));
                            }
                            stringBuilder.append(",");
                        } catch (InvocationTargetException | IllegalAccessException e) {
                            logger.error("csv export error", e);
                        }
                    });
                    stringBuilder.deleteCharAt(stringBuilder.length() - 1);
                    fun.apply(null);
                }
        );

        return stringBuilder.toString().getBytes(StandardCharsets.UTF_8);
    }
}