package com.wlos.app.bl;

import com.wlos.app.exception.BusinessException;
import com.wlos.app.utils.Result;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.wlos.app.utils.Constants;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;


import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;


import java.util.HashMap;
import java.util.Base64;
import com.alibaba.fastjson.JSON;


@Service
public class ImportService implements ApplicationContextAware {
    private final Logger log = LoggerFactory.getLogger(ImportService.class);

    private ApplicationContext applicationContext;

    private final Map<String, ProgressStatus> importResult = new ConcurrentHashMap<>();

    private final List<ParseService> parseServiceList;

    private static final String DOWNLOAD_FILE_PATH = "/tmp/";

    public ImportService(ObjectProvider<ParseService> objectProvider) {
        this.parseServiceList = objectProvider.stream().collect(Collectors.toList());
    }

    public void importTemplate(String fileType, String model, HttpServletResponse servletResponse) throws Exception {
        ParseService parseService = getParseService(fileType);
        String fileName = "Template_" + model + "." + fileType;
        String finalModel = "com.wlos.app.model." + model;
        response(servletResponse, parseService.importTemplate((Class<Object>) Class.forName(finalModel)), fileName);
    }

    public ProgressStatus checkStatus(String key) {
        return importResult.get(key);
    }

    public Object importModel(String fileType, String model, byte[] file) throws Exception {
        String importKey = DigestUtils.md5Hex(file);
        if (importResult.containsKey(importKey)) {
            return importResult.get(importKey);
        }

        importResult.put(importKey, new ProgressStatus());
        String finalModel = "com.wlos.app.model." + model;
        ParseService parseService = getParseService(fileType);

        String serviceName = model + "Service";
        Class serviceClass = Class.forName("com.wlos.app.da.service." + serviceName);
        Object entityService = applicationContext.getBean(serviceClass);
        Method insertMethod = serviceClass.getMethod("insert", Class.forName(finalModel));

        new Thread(new ImportTask(parseService, entityService, insertMethod, (Class<Object>) Class.forName(finalModel), file, importResult.get(importKey))).start();

        return importKey;
    }

    public String exportModel(String fileType, String model, Map<String, Object> condition, List<String> headers) throws Exception {
        StringBuilder key = new StringBuilder(fileType);
        key.append("_")
                .append(model)
                .append("_");
        condition.forEach((k, v) ->
                key.append(k)
                        .append("=")
                        .append(v)
        );
        String exportKey = model + "_" + DigestUtils.md5Hex(key.toString()) + "." + fileType;
        log.debug("exportModel key = {} sum ={}", exportKey, key);
        importResult.put(exportKey, new ProgressStatus(DOWNLOAD_FILE_PATH, exportKey));

        String finalModel = "com.wlos.app.model." + model;
        ParseService parseService = getParseService(fileType);

        String serviceName = model + "Service";
        Class serviceClass = Class.forName("com.wlos.app.da.service." + serviceName);
        Object entityService = applicationContext.getBean(serviceClass);
        Method queryAllMethod = serviceClass.getMethod("queryAll", Class.forName(finalModel),int.class,int.class);

        Object entityInstance = Class.forName(finalModel).getDeclaredConstructor().newInstance();
        BeanUtils.copyProperties(condition, entityInstance);

        new Thread(
                new ExportTask(parseService, entityService, queryAllMethod, (Class<Object>) Class.forName(finalModel), entityInstance, headers, importResult.get(exportKey)))
                .start();

        return exportKey;
    }

    public void checkExport(String key, HttpServletResponse servletResponse) throws IOException {
        if (importResult.containsKey(key) && importResult.get(key).getPercent() == 100) {
            response(servletResponse, Files.readAllBytes(Path.of(DOWNLOAD_FILE_PATH + key)), key);
        } else {
            throw new BusinessException("export not finish");
        }
    }

    static class ImportTask implements Runnable {
        private final ParseService parseService;
        private final Object entityService;
        private final Method method;
        private final Class<Object> clazz;
        private final byte[] file;
        private final ProgressStatus status;
        private final Logger log = LoggerFactory.getLogger(getClass());

        ImportTask(ParseService parseService, Object entityService, Method method, Class<Object> clazz, byte[] file, ProgressStatus status) {
            this.parseService = parseService;
            this.entityService = entityService;
            this.method = method;
            this.clazz = clazz;
            this.file = file;
            this.status = status;
        }

        @Override
        public void run() {
            try {
                List<Object> list = parseService.parse(clazz, file);
                status.total = list.size();
                list.stream().map(clazz::cast)
                        .forEach(it -> {
                            try {
                                method.invoke(entityService, it);
                                status.index++;
                                log.debug("import success {} ", it);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        });
            } catch (Exception e) {
                log.error("import error", e);
                status.msg = e + ":" + e.getMessage();
            }
        }
    }

    static class ExportTask implements Runnable {
        private final ParseService parseService;
        private final Object entityService;
        private final Method method;
        private final Class<Object> clazz;
        private final Object entityInstance;
        private final List<String> headers;
        private final ProgressStatus progressStatus;
        private final Logger log = LoggerFactory.getLogger(getClass());

        ExportTask(ParseService parseService, Object entityService, Method method, Class<Object> clazz, Object entityInstance, List<String> headers, ProgressStatus progressStatus) {
            this.parseService = parseService;
            this.entityService = entityService;
            this.method = method;
            this.clazz = clazz;
            this.entityInstance = entityInstance;
            this.headers = headers;
            this.progressStatus = progressStatus;
        }

        @Override
        public void run() {
            Map<String, Object> map;
            try {
                map = (Map<String, Object>) method.invoke(entityService, entityInstance,1, Integer.MAX_VALUE);
                List<Object> list = (List<Object>) map.get(Constants.RESULT_DATA);
                progressStatus.total = list.size();
                byte[] result = parseService.exportModel(clazz, list, headers, progressStatus::addIndex);
                File temp = new File(progressStatus.filePath, progressStatus.fileName);
                temp.deleteOnExit();
                Files.write(temp.toPath(), result);
            } catch (IllegalAccessException | InvocationTargetException | IOException e) {
                log.error("export error", e);
                progressStatus.msg = e + ":" + e.getMessage();
            }
        }
    }

    static class ProgressStatus {
        Integer total = -1;
        Integer index = 0;
        Integer percent = -1;
        String msg = "";
        String fileName = "";
        String filePath = "";

        public ProgressStatus() {
        }

        public ProgressStatus(String filePath, String fileName) {
            this.fileName = fileName;
            this.filePath = filePath;
        }

        public String getDownloadUrl() {
            if (StringUtils.isNotBlank(fileName)) {
                return "/downloadFile/" + fileName;
            }
            return fileName;
        }

        public Integer getPercent() {

            if (total > 0 && index > 0) {
                return (index * 100) / total;
            }

            if (total > 0) {
                return 0;
            }

            return percent;
        }

        @Override
        public String toString() {
            return "{\"total\":" + total + ",\"completed\":" + index + ",\"percent\":" + getPercent() + ",\"fileUrl\":\"" + getDownloadUrl() + "\"" + ",\"errmsg\":\"" + msg + "\"}";
        }

        public Object addIndex(Object o) {
            index++;
            return null;
        }
    }


    private void response(HttpServletResponse servletResponse, byte[] file, String fileName) {
        //返回base64
        Map<String, Object> result = new HashMap<>();
        result.put("fileName",fileName);
        result.put("bytes",Base64.getEncoder().encodeToString(file));
        byte[] resultBytes = JSON.toJSONString(Result.SUCCESS(result)).getBytes(StandardCharsets.UTF_8);
        try(
            OutputStream out = servletResponse.getOutputStream()
        ){
            servletResponse.setContentType("application/json; charset=utf-8");
            servletResponse.setContentLength(resultBytes.length);
            out.write(resultBytes);
            out.flush();
        }catch (Exception e){
            log.error("download err", e);
        }

        /*//返回流
        try (
            InputStream inputStream = new ByteArrayInputStream(file);
            OutputStream out = servletResponse.getOutputStream()
        ) {
            servletResponse.setCharacterEncoding("utf-8");
            servletResponse.setContentType("application/octet-stream;charset=utf-8");
            servletResponse.setHeader("Content-Disposition", "attachment; fileName=" + URLEncoder.encode(fileName, StandardCharsets.UTF_8));
            byte[] b = new byte[1024];  //创建数据缓冲区
            int length;
            while ((length = inputStream.read(b)) > 0) {
                out.write(b, 0, length);
            }
            out.flush();
        } catch (IOException e) {
            log.error("download err", e);
        }*/
    }

    private ParseService getParseService(String fileType) {
        return parseServiceList.stream()
                .filter(it -> it.support(fileType))
                .findFirst()
                .orElseThrow(() -> new BusinessException("不支持的格式"));
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}