package com.wlos.app.api;

import com.wlos.app.model.VenueInformation;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.VenueInformationService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 场馆信息服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:35 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class VenueInformationController {

    @Autowired
    private VenueInformationService venueInformationService;


    /**
     * 根据id查询场馆信息
     *
     * @param id 场馆信息Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/venueInformation/{id}")
    public Result<VenueInformation> getById(@PathVariable String id) {
        VenueInformation venueInformation = venueInformationService.queryById(id);
        return Result.SUCCESS(venueInformation);
    }

    /**
     * 根据id列表查询场馆信息
     *
     * @param ids 场馆信息Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/venueInformation/batch")
    public Result<List<VenueInformation>> getByIds(@RequestBody List<String> ids) {
        List<VenueInformation> venueInformationList = venueInformationService.listByIds(ids);
        return Result.SUCCESS(venueInformationList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/venueInformation/merge")
    public Result<Set<VenueInformation>> merge(@RequestBody List<List<VenueInformation>> list) {
        Set<VenueInformation> newMergedList = new HashSet<>();

        for (List<VenueInformation> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/venueInformation/extract")
    public Result<List<String>> extract(@RequestBody List<VenueInformation> list) {
        List<String> idList = list.stream().map(VenueInformation::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询场馆信息
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/venueInformation/one")
    public Result<VenueInformation> one(VenueInformation condition) {
        VenueInformation venueInformation = venueInformationService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(venueInformation);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/venueInformation/count")
    public Result<Long> count(VenueInformation condition) {
        long count = venueInformationService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/venueInformation/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, VenueInformation condition) {
        QueryWrapper<VenueInformation> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = venueInformationService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/venueInformation/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<VenueInformation> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = venueInformationService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 场馆信息
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/venueInformation/page")
    public Result<Page<VenueInformation>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            VenueInformation condition) {
        Page<VenueInformation> pageResult = venueInformationService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 场馆信息
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/venueInformation/list")
    public Result<List<VenueInformation>> list(VenueInformation condition) {
        List<VenueInformation> listResult = venueInformationService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/venueInformation/condition/list")
    public Result<List<VenueInformation>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<VenueInformation> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<VenueInformation> list = venueInformationService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建场馆信息
     *
     * @param venueInformation 场馆信息载体
     * @return 场馆信息 主键
     */
    @PostMapping("/v1/venueInformation")
    public Result<String> create(@RequestBody VenueInformation venueInformation){
        return Result.SUCCESS(venueInformationService.insert(venueInformation));
    }


    /**
     * 根据主键更新场馆信息
     *
     * @param id 场馆信息ID
     * @param venueInformation 场馆信息载体
     * @return Id
     */
    @PutMapping("/v1/venueInformation/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody VenueInformation venueInformation){
        return Result.SUCCESS(venueInformationService.update(id, venueInformation));
    }


    /**
     * 根据主键删除场馆信息
     *
     * @param id 场馆信息ID
     * @return
     */
    @DeleteMapping("/v1/venueInformation/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        venueInformationService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除场馆信息
     *
     * @param ids 场馆信息id集合
     * @return
     */
    @DeleteMapping("/v1/venueInformation/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        venueInformationService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}