package com.wlos.app.api;

import com.wlos.app.model.SurveyTask;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyTaskService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 任务信息服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:34 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyTaskController {

    @Autowired
    private SurveyTaskService surveyTaskService;


    /**
     * 根据id查询任务信息
     *
     * @param id 任务信息Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyTask/{id}")
    public Result<SurveyTask> getById(@PathVariable String id) {
        SurveyTask surveyTask = surveyTaskService.queryById(id);
        return Result.SUCCESS(surveyTask);
    }

    /**
     * 根据id列表查询任务信息
     *
     * @param ids 任务信息Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyTask/batch")
    public Result<List<SurveyTask>> getByIds(@RequestBody List<String> ids) {
        List<SurveyTask> surveyTaskList = surveyTaskService.listByIds(ids);
        return Result.SUCCESS(surveyTaskList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyTask/merge")
    public Result<Set<SurveyTask>> merge(@RequestBody List<List<SurveyTask>> list) {
        Set<SurveyTask> newMergedList = new HashSet<>();

        for (List<SurveyTask> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyTask/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyTask> list) {
        List<String> idList = list.stream().map(SurveyTask::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询任务信息
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyTask/one")
    public Result<SurveyTask> one(SurveyTask condition) {
        SurveyTask surveyTask = surveyTaskService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyTask);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyTask/count")
    public Result<Long> count(SurveyTask condition) {
        long count = surveyTaskService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyTask/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyTask condition) {
        QueryWrapper<SurveyTask> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyTaskService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyTask/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyTask> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyTaskService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 任务信息
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyTask/page")
    public Result<Page<SurveyTask>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyTask condition) {
        Page<SurveyTask> pageResult = surveyTaskService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 任务信息
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyTask/list")
    public Result<List<SurveyTask>> list(SurveyTask condition) {
        List<SurveyTask> listResult = surveyTaskService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyTask/condition/list")
    public Result<List<SurveyTask>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyTask> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyTask> list = surveyTaskService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建任务信息
     *
     * @param surveyTask 任务信息载体
     * @return 任务信息 主键
     */
    @PostMapping("/v1/surveyTask")
    public Result<String> create(@RequestBody SurveyTask surveyTask){
        return Result.SUCCESS(surveyTaskService.insert(surveyTask));
    }


    /**
     * 根据主键更新任务信息
     *
     * @param id 任务信息ID
     * @param surveyTask 任务信息载体
     * @return Id
     */
    @PutMapping("/v1/surveyTask/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyTask surveyTask){
        return Result.SUCCESS(surveyTaskService.update(id, surveyTask));
    }


    /**
     * 根据主键删除任务信息
     *
     * @param id 任务信息ID
     * @return
     */
    @DeleteMapping("/v1/surveyTask/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyTaskService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除任务信息
     *
     * @param ids 任务信息id集合
     * @return
     */
    @DeleteMapping("/v1/surveyTask/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyTaskService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}