package com.wlos.app.api;

import com.wlos.app.model.SurveySign;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveySignService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 勘查签到表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveySignController {

    @Autowired
    private SurveySignService surveySignService;


    /**
     * 根据id查询勘查签到表
     *
     * @param id 勘查签到表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveySign/{id}")
    public Result<SurveySign> getById(@PathVariable String id) {
        SurveySign surveySign = surveySignService.queryById(id);
        return Result.SUCCESS(surveySign);
    }

    /**
     * 根据id列表查询勘查签到表
     *
     * @param ids 勘查签到表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveySign/batch")
    public Result<List<SurveySign>> getByIds(@RequestBody List<String> ids) {
        List<SurveySign> surveySignList = surveySignService.listByIds(ids);
        return Result.SUCCESS(surveySignList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveySign/merge")
    public Result<Set<SurveySign>> merge(@RequestBody List<List<SurveySign>> list) {
        Set<SurveySign> newMergedList = new HashSet<>();

        for (List<SurveySign> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveySign/extract")
    public Result<List<String>> extract(@RequestBody List<SurveySign> list) {
        List<String> idList = list.stream().map(SurveySign::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询勘查签到表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveySign/one")
    public Result<SurveySign> one(SurveySign condition) {
        SurveySign surveySign = surveySignService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveySign);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveySign/count")
    public Result<Long> count(SurveySign condition) {
        long count = surveySignService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveySign/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveySign condition) {
        QueryWrapper<SurveySign> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveySignService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveySign/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveySign> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveySignService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 勘查签到表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveySign/page")
    public Result<Page<SurveySign>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveySign condition) {
        Page<SurveySign> pageResult = surveySignService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 勘查签到表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveySign/list")
    public Result<List<SurveySign>> list(SurveySign condition) {
        List<SurveySign> listResult = surveySignService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveySign/condition/list")
    public Result<List<SurveySign>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveySign> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveySign> list = surveySignService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建勘查签到表
     *
     * @param surveySign 勘查签到表载体
     * @return 勘查签到表 主键
     */
    @PostMapping("/v1/surveySign")
    public Result<String> create(@RequestBody SurveySign surveySign){
        return Result.SUCCESS(surveySignService.insert(surveySign));
    }


    /**
     * 根据主键更新勘查签到表
     *
     * @param id 勘查签到表ID
     * @param surveySign 勘查签到表载体
     * @return Id
     */
    @PutMapping("/v1/surveySign/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveySign surveySign){
        return Result.SUCCESS(surveySignService.update(id, surveySign));
    }


    /**
     * 根据主键删除勘查签到表
     *
     * @param id 勘查签到表ID
     * @return
     */
    @DeleteMapping("/v1/surveySign/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveySignService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除勘查签到表
     *
     * @param ids 勘查签到表id集合
     * @return
     */
    @DeleteMapping("/v1/surveySign/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveySignService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}