package com.wlos.app.api;

import com.wlos.app.model.SurveyReform;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyReformService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 回复整改记录表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:25 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyReformController {

    @Autowired
    private SurveyReformService surveyReformService;


    /**
     * 根据id查询回复整改记录表
     *
     * @param id 回复整改记录表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyReform/{id}")
    public Result<SurveyReform> getById(@PathVariable String id) {
        SurveyReform surveyReform = surveyReformService.queryById(id);
        return Result.SUCCESS(surveyReform);
    }

    /**
     * 根据id列表查询回复整改记录表
     *
     * @param ids 回复整改记录表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyReform/batch")
    public Result<List<SurveyReform>> getByIds(@RequestBody List<String> ids) {
        List<SurveyReform> surveyReformList = surveyReformService.listByIds(ids);
        return Result.SUCCESS(surveyReformList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyReform/merge")
    public Result<Set<SurveyReform>> merge(@RequestBody List<List<SurveyReform>> list) {
        Set<SurveyReform> newMergedList = new HashSet<>();

        for (List<SurveyReform> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyReform/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyReform> list) {
        List<String> idList = list.stream().map(SurveyReform::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询回复整改记录表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyReform/one")
    public Result<SurveyReform> one(SurveyReform condition) {
        SurveyReform surveyReform = surveyReformService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyReform);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyReform/count")
    public Result<Long> count(SurveyReform condition) {
        long count = surveyReformService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyReform/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyReform condition) {
        QueryWrapper<SurveyReform> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyReformService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyReform/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyReform> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyReformService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 回复整改记录表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyReform/page")
    public Result<Page<SurveyReform>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyReform condition) {
        Page<SurveyReform> pageResult = surveyReformService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 回复整改记录表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyReform/list")
    public Result<List<SurveyReform>> list(SurveyReform condition) {
        List<SurveyReform> listResult = surveyReformService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyReform/condition/list")
    public Result<List<SurveyReform>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyReform> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyReform> list = surveyReformService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建回复整改记录表
     *
     * @param surveyReform 回复整改记录表载体
     * @return 回复整改记录表 主键
     */
    @PostMapping("/v1/surveyReform")
    public Result<String> create(@RequestBody SurveyReform surveyReform){
        return Result.SUCCESS(surveyReformService.insert(surveyReform));
    }


    /**
     * 根据主键更新回复整改记录表
     *
     * @param id 回复整改记录表ID
     * @param surveyReform 回复整改记录表载体
     * @return Id
     */
    @PutMapping("/v1/surveyReform/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyReform surveyReform){
        return Result.SUCCESS(surveyReformService.update(id, surveyReform));
    }


    /**
     * 根据主键删除回复整改记录表
     *
     * @param id 回复整改记录表ID
     * @return
     */
    @DeleteMapping("/v1/surveyReform/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyReformService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除回复整改记录表
     *
     * @param ids 回复整改记录表id集合
     * @return
     */
    @DeleteMapping("/v1/surveyReform/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyReformService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}