package com.wlos.app.api;

import com.wlos.app.model.SurveyQuestionnaire;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyQuestionnaireService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 问卷服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:27 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyQuestionnaireController {

    @Autowired
    private SurveyQuestionnaireService surveyQuestionnaireService;


    /**
     * 根据id查询问卷
     *
     * @param id 问卷Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyQuestionnaire/{id}")
    public Result<SurveyQuestionnaire> getById(@PathVariable String id) {
        SurveyQuestionnaire surveyQuestionnaire = surveyQuestionnaireService.queryById(id);
        return Result.SUCCESS(surveyQuestionnaire);
    }

    /**
     * 根据id列表查询问卷
     *
     * @param ids 问卷Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyQuestionnaire/batch")
    public Result<List<SurveyQuestionnaire>> getByIds(@RequestBody List<String> ids) {
        List<SurveyQuestionnaire> surveyQuestionnaireList = surveyQuestionnaireService.listByIds(ids);
        return Result.SUCCESS(surveyQuestionnaireList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyQuestionnaire/merge")
    public Result<Set<SurveyQuestionnaire>> merge(@RequestBody List<List<SurveyQuestionnaire>> list) {
        Set<SurveyQuestionnaire> newMergedList = new HashSet<>();

        for (List<SurveyQuestionnaire> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyQuestionnaire/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyQuestionnaire> list) {
        List<String> idList = list.stream().map(SurveyQuestionnaire::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询问卷
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyQuestionnaire/one")
    public Result<SurveyQuestionnaire> one(SurveyQuestionnaire condition) {
        SurveyQuestionnaire surveyQuestionnaire = surveyQuestionnaireService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyQuestionnaire);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyQuestionnaire/count")
    public Result<Long> count(SurveyQuestionnaire condition) {
        long count = surveyQuestionnaireService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyQuestionnaire/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyQuestionnaire condition) {
        QueryWrapper<SurveyQuestionnaire> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyQuestionnaireService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyQuestionnaire/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyQuestionnaire> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyQuestionnaireService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 问卷
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyQuestionnaire/page")
    public Result<Page<SurveyQuestionnaire>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyQuestionnaire condition) {
        Page<SurveyQuestionnaire> pageResult = surveyQuestionnaireService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 问卷
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyQuestionnaire/list")
    public Result<List<SurveyQuestionnaire>> list(SurveyQuestionnaire condition) {
        List<SurveyQuestionnaire> listResult = surveyQuestionnaireService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyQuestionnaire/condition/list")
    public Result<List<SurveyQuestionnaire>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyQuestionnaire> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyQuestionnaire> list = surveyQuestionnaireService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建问卷
     *
     * @param surveyQuestionnaire 问卷载体
     * @return 问卷 主键
     */
    @PostMapping("/v1/surveyQuestionnaire")
    public Result<String> create(@RequestBody SurveyQuestionnaire surveyQuestionnaire){
        return Result.SUCCESS(surveyQuestionnaireService.insert(surveyQuestionnaire));
    }


    /**
     * 根据主键更新问卷
     *
     * @param id 问卷ID
     * @param surveyQuestionnaire 问卷载体
     * @return Id
     */
    @PutMapping("/v1/surveyQuestionnaire/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyQuestionnaire surveyQuestionnaire){
        return Result.SUCCESS(surveyQuestionnaireService.update(id, surveyQuestionnaire));
    }


    /**
     * 根据主键删除问卷
     *
     * @param id 问卷ID
     * @return
     */
    @DeleteMapping("/v1/surveyQuestionnaire/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyQuestionnaireService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除问卷
     *
     * @param ids 问卷id集合
     * @return
     */
    @DeleteMapping("/v1/surveyQuestionnaire/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyQuestionnaireService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}