package com.wlos.app.api;

import com.wlos.app.model.SurveyProject;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyProjectService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 勘查项目服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyProjectController {

    @Autowired
    private SurveyProjectService surveyProjectService;


    /**
     * 根据id查询勘查项目
     *
     * @param id 勘查项目Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyProject/{id}")
    public Result<SurveyProject> getById(@PathVariable String id) {
        SurveyProject surveyProject = surveyProjectService.queryById(id);
        return Result.SUCCESS(surveyProject);
    }

    /**
     * 根据id列表查询勘查项目
     *
     * @param ids 勘查项目Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyProject/batch")
    public Result<List<SurveyProject>> getByIds(@RequestBody List<String> ids) {
        List<SurveyProject> surveyProjectList = surveyProjectService.listByIds(ids);
        return Result.SUCCESS(surveyProjectList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyProject/merge")
    public Result<Set<SurveyProject>> merge(@RequestBody List<List<SurveyProject>> list) {
        Set<SurveyProject> newMergedList = new HashSet<>();

        for (List<SurveyProject> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyProject/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyProject> list) {
        List<String> idList = list.stream().map(SurveyProject::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询勘查项目
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyProject/one")
    public Result<SurveyProject> one(SurveyProject condition) {
        SurveyProject surveyProject = surveyProjectService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyProject);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyProject/count")
    public Result<Long> count(SurveyProject condition) {
        long count = surveyProjectService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyProject/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyProject condition) {
        QueryWrapper<SurveyProject> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyProjectService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyProject/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyProject> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyProjectService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 勘查项目
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyProject/page")
    public Result<Page<SurveyProject>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyProject condition) {
        Page<SurveyProject> pageResult = surveyProjectService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 勘查项目
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyProject/list")
    public Result<List<SurveyProject>> list(SurveyProject condition) {
        List<SurveyProject> listResult = surveyProjectService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyProject/condition/list")
    public Result<List<SurveyProject>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyProject> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyProject> list = surveyProjectService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建勘查项目
     *
     * @param surveyProject 勘查项目载体
     * @return 勘查项目 主键
     */
    @PostMapping("/v1/surveyProject")
    public Result<String> create(@RequestBody SurveyProject surveyProject){
        return Result.SUCCESS(surveyProjectService.insert(surveyProject));
    }


    /**
     * 根据主键更新勘查项目
     *
     * @param id 勘查项目ID
     * @param surveyProject 勘查项目载体
     * @return Id
     */
    @PutMapping("/v1/surveyProject/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyProject surveyProject){
        return Result.SUCCESS(surveyProjectService.update(id, surveyProject));
    }


    /**
     * 根据主键删除勘查项目
     *
     * @param id 勘查项目ID
     * @return
     */
    @DeleteMapping("/v1/surveyProject/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyProjectService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除勘查项目
     *
     * @param ids 勘查项目id集合
     * @return
     */
    @DeleteMapping("/v1/surveyProject/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyProjectService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}