package com.wlos.app.api;

import com.wlos.app.model.SurveyPhoto;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyPhotoService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 查勘现场照片服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyPhotoController {

    @Autowired
    private SurveyPhotoService surveyPhotoService;


    /**
     * 根据id查询查勘现场照片
     *
     * @param id 查勘现场照片Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyPhoto/{id}")
    public Result<SurveyPhoto> getById(@PathVariable String id) {
        SurveyPhoto surveyPhoto = surveyPhotoService.queryById(id);
        return Result.SUCCESS(surveyPhoto);
    }

    /**
     * 根据id列表查询查勘现场照片
     *
     * @param ids 查勘现场照片Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyPhoto/batch")
    public Result<List<SurveyPhoto>> getByIds(@RequestBody List<String> ids) {
        List<SurveyPhoto> surveyPhotoList = surveyPhotoService.listByIds(ids);
        return Result.SUCCESS(surveyPhotoList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyPhoto/merge")
    public Result<Set<SurveyPhoto>> merge(@RequestBody List<List<SurveyPhoto>> list) {
        Set<SurveyPhoto> newMergedList = new HashSet<>();

        for (List<SurveyPhoto> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyPhoto/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyPhoto> list) {
        List<String> idList = list.stream().map(SurveyPhoto::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询查勘现场照片
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyPhoto/one")
    public Result<SurveyPhoto> one(SurveyPhoto condition) {
        SurveyPhoto surveyPhoto = surveyPhotoService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyPhoto);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyPhoto/count")
    public Result<Long> count(SurveyPhoto condition) {
        long count = surveyPhotoService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyPhoto/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyPhoto condition) {
        QueryWrapper<SurveyPhoto> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyPhotoService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyPhoto/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyPhoto> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyPhotoService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 查勘现场照片
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyPhoto/page")
    public Result<Page<SurveyPhoto>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyPhoto condition) {
        Page<SurveyPhoto> pageResult = surveyPhotoService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 查勘现场照片
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyPhoto/list")
    public Result<List<SurveyPhoto>> list(SurveyPhoto condition) {
        List<SurveyPhoto> listResult = surveyPhotoService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyPhoto/condition/list")
    public Result<List<SurveyPhoto>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyPhoto> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyPhoto> list = surveyPhotoService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建查勘现场照片
     *
     * @param surveyPhoto 查勘现场照片载体
     * @return 查勘现场照片 主键
     */
    @PostMapping("/v1/surveyPhoto")
    public Result<String> create(@RequestBody SurveyPhoto surveyPhoto){
        return Result.SUCCESS(surveyPhotoService.insert(surveyPhoto));
    }


    /**
     * 根据主键更新查勘现场照片
     *
     * @param id 查勘现场照片ID
     * @param surveyPhoto 查勘现场照片载体
     * @return Id
     */
    @PutMapping("/v1/surveyPhoto/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyPhoto surveyPhoto){
        return Result.SUCCESS(surveyPhotoService.update(id, surveyPhoto));
    }


    /**
     * 根据主键删除查勘现场照片
     *
     * @param id 查勘现场照片ID
     * @return
     */
    @DeleteMapping("/v1/surveyPhoto/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyPhotoService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除查勘现场照片
     *
     * @param ids 查勘现场照片id集合
     * @return
     */
    @DeleteMapping("/v1/surveyPhoto/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyPhotoService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}