package com.wlos.app.api;

import com.wlos.app.model.SurveyExpert;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyExpertService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 踏勘专家记录表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyExpertController {

    @Autowired
    private SurveyExpertService surveyExpertService;


    /**
     * 根据id查询踏勘专家记录表
     *
     * @param id 踏勘专家记录表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyExpert/{id}")
    public Result<SurveyExpert> getById(@PathVariable String id) {
        SurveyExpert surveyExpert = surveyExpertService.queryById(id);
        return Result.SUCCESS(surveyExpert);
    }

    /**
     * 根据id列表查询踏勘专家记录表
     *
     * @param ids 踏勘专家记录表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyExpert/batch")
    public Result<List<SurveyExpert>> getByIds(@RequestBody List<String> ids) {
        List<SurveyExpert> surveyExpertList = surveyExpertService.listByIds(ids);
        return Result.SUCCESS(surveyExpertList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyExpert/merge")
    public Result<Set<SurveyExpert>> merge(@RequestBody List<List<SurveyExpert>> list) {
        Set<SurveyExpert> newMergedList = new HashSet<>();

        for (List<SurveyExpert> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyExpert/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyExpert> list) {
        List<String> idList = list.stream().map(SurveyExpert::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询踏勘专家记录表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyExpert/one")
    public Result<SurveyExpert> one(SurveyExpert condition) {
        SurveyExpert surveyExpert = surveyExpertService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyExpert);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyExpert/count")
    public Result<Long> count(SurveyExpert condition) {
        long count = surveyExpertService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyExpert/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyExpert condition) {
        QueryWrapper<SurveyExpert> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyExpertService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyExpert/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyExpert> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyExpertService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 踏勘专家记录表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyExpert/page")
    public Result<Page<SurveyExpert>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyExpert condition) {
        Page<SurveyExpert> pageResult = surveyExpertService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 踏勘专家记录表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyExpert/list")
    public Result<List<SurveyExpert>> list(SurveyExpert condition) {
        List<SurveyExpert> listResult = surveyExpertService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyExpert/condition/list")
    public Result<List<SurveyExpert>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyExpert> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyExpert> list = surveyExpertService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建踏勘专家记录表
     *
     * @param surveyExpert 踏勘专家记录表载体
     * @return 踏勘专家记录表 主键
     */
    @PostMapping("/v1/surveyExpert")
    public Result<String> create(@RequestBody SurveyExpert surveyExpert){
        return Result.SUCCESS(surveyExpertService.insert(surveyExpert));
    }


    /**
     * 根据主键更新踏勘专家记录表
     *
     * @param id 踏勘专家记录表ID
     * @param surveyExpert 踏勘专家记录表载体
     * @return Id
     */
    @PutMapping("/v1/surveyExpert/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyExpert surveyExpert){
        return Result.SUCCESS(surveyExpertService.update(id, surveyExpert));
    }


    /**
     * 根据主键删除踏勘专家记录表
     *
     * @param id 踏勘专家记录表ID
     * @return
     */
    @DeleteMapping("/v1/surveyExpert/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyExpertService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除踏勘专家记录表
     *
     * @param ids 踏勘专家记录表id集合
     * @return
     */
    @DeleteMapping("/v1/surveyExpert/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyExpertService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}