package com.wlos.app.api;

import com.wlos.app.bl.ImportService;
import com.wlos.app.utils.Constants;
import com.wlos.app.utils.Result;
import com.alibaba.fastjson.JSON;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Locale;


/**
 * 文件导入导出
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class ImportController {

    private final ImportService importService;

    public ImportController(ImportService importService) {
        this.importService = importService;
    }

    /**
     * 获取数据导入模版
     *
     * @param fileType 导入文件类型
     * @param model    导入的模型名称
     */
    @GetMapping("/importTemplate")
    public void importTemplate(@RequestParam String fileType, @RequestParam String model, HttpServletResponse servletResponse) throws Exception {
        model = transModelName(model);
        importService.importTemplate(fileType, model, servletResponse);
    }

    /**
     * 查询导入导出结果
     *
     * @param key 文件标识符
     * @return 处理进度
     */
    @GetMapping("/checkStatus")
    public Result checkStatus(@RequestParam String key) {
        return Result.SUCCESS(importService.checkStatus(key));
    }

    /**
     * 数据导入 相同文件不可重复提交
     *
     * @param fileType 文件类型
     * @param model    导入的模型名称
     * @param file     数据文件
     * @return 导入文件标识符，调用checkStatus接口查询导入进度
     */
    @PostMapping("/import")
    public Result importModel(@RequestParam String fileType, @RequestParam String model, @RequestParam MultipartFile file) throws Exception {
        model = transModelName(model);
        return Result.SUCCESS(importService.importModel(fileType, model, file.getBytes()));
    }


    /**
     * 数据导出 同一入参只会串行生成
     *
     * @param fileType 文件类型 json,xlsx,csv
     * @param model    导入的模型名称
     * @param headers  需要导出的字段名称 逗号分割
     * @return 导出文件标识符，调用checkStatus接口查询导出进度
     */
    @PostMapping("/export")
    public Result exportModel(@RequestParam String fileType,
                              @RequestParam String model,
                              @RequestParam(required = false, defaultValue = "") String headers,
                              @RequestParam(required = false, defaultValue = "{}") String condition) throws Exception {
        model = transModelName(model);
        return Result.SUCCESS(importService.exportModel(fileType, model, JSON.parseObject(condition), List.of(headers.split(","))));
    }

    /**
     * 根据文件标识符下载文件
     * @param key 文件标识符
     */
    @GetMapping("/downloadFile/{key}")
    public void downloadFile(@PathVariable String key, HttpServletResponse servletResponse) throws IOException {
        importService.checkExport(key, servletResponse);
    }

    private String transModelName(String para) {

        char[] cs=para.toCharArray();
        cs[0]-=32;
        return String.valueOf(cs);

        /* para = para.toLowerCase(Locale.ROOT);
        StringBuilder result = new StringBuilder();
        String[] a = para.split("_");
        if (a.length > 1) {
            for (String s : a) {
                result.append(s.substring(0, 1).toUpperCase());
                result.append(s.substring(1).toLowerCase());
            }
            return result.toString();
        } else {
            return para.substring(0, 1).toUpperCase() + para.substring(1);
        }*/
    }
}
