package com.wlos.app.api;

import com.wlos.app.model.BaseDict;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.BaseDictService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 编码表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:24 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class BaseDictController {

    @Autowired
    private BaseDictService baseDictService;


    /**
     * 根据id查询编码表
     *
     * @param id 编码表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseDict/{id}")
    public Result<BaseDict> getById(@PathVariable String id) {
        BaseDict baseDict = baseDictService.queryById(id);
        return Result.SUCCESS(baseDict);
    }

    /**
     * 根据id列表查询编码表
     *
     * @param ids 编码表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/baseDict/batch")
    public Result<List<BaseDict>> getByIds(@RequestBody List<String> ids) {
        List<BaseDict> baseDictList = baseDictService.listByIds(ids);
        return Result.SUCCESS(baseDictList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/baseDict/merge")
    public Result<Set<BaseDict>> merge(@RequestBody List<List<BaseDict>> list) {
        Set<BaseDict> newMergedList = new HashSet<>();

        for (List<BaseDict> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/baseDict/extract")
    public Result<List<String>> extract(@RequestBody List<BaseDict> list) {
        List<String> idList = list.stream().map(BaseDict::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询编码表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseDict/one")
    public Result<BaseDict> one(BaseDict condition) {
        BaseDict baseDict = baseDictService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(baseDict);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/baseDict/count")
    public Result<Long> count(BaseDict condition) {
        long count = baseDictService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/baseDict/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, BaseDict condition) {
        QueryWrapper<BaseDict> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = baseDictService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/baseDict/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<BaseDict> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = baseDictService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 编码表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseDict/page")
    public Result<Page<BaseDict>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            BaseDict condition) {
        Page<BaseDict> pageResult = baseDictService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 编码表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseDict/list")
    public Result<List<BaseDict>> list(BaseDict condition) {
        List<BaseDict> listResult = baseDictService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/baseDict/condition/list")
    public Result<List<BaseDict>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<BaseDict> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<BaseDict> list = baseDictService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建编码表
     *
     * @param baseDict 编码表载体
     * @return 编码表 主键
     */
    @PostMapping("/v1/baseDict")
    public Result<String> create(@RequestBody BaseDict baseDict){
        return Result.SUCCESS(baseDictService.insert(baseDict));
    }


    /**
     * 根据主键更新编码表
     *
     * @param id 编码表ID
     * @param baseDict 编码表载体
     * @return Id
     */
    @PutMapping("/v1/baseDict/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody BaseDict baseDict){
        return Result.SUCCESS(baseDictService.update(id, baseDict));
    }


    /**
     * 根据主键删除编码表
     *
     * @param id 编码表ID
     * @return
     */
    @DeleteMapping("/v1/baseDict/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        baseDictService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除编码表
     *
     * @param ids 编码表id集合
     * @return
     */
    @DeleteMapping("/v1/baseDict/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        baseDictService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}