package com.bcxin.fdd.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Validator;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.fdd.common.emus.TaskStatusEnum;
import com.bcxin.fdd.common.emus.TaskTypeEnum;
import com.bcxin.fdd.common.exception.AssertException;
import com.bcxin.fdd.common.utils.*;
import com.bcxin.fdd.configs.FddConfig;
import com.bcxin.fdd.dao.mapper.BfOrganizationMapper;
import com.bcxin.fdd.dao.mapper.BfSignTaskMapper;
import com.bcxin.fdd.dtos.request.SignTaskActorGetUrlRequest;
import com.bcxin.fdd.dtos.request.SignTaskCreateRequest;
import com.bcxin.fdd.dtos.response.BcxTenantUserResponse;
import com.bcxin.fdd.dtos.response.GetFddUrlResponse;
import com.bcxin.fdd.entity.BfOrganizationEntity;
import com.bcxin.fdd.entity.BfSignTaskEntity;
import com.fasc.open.api.bean.base.BaseRes;
import com.fasc.open.api.bean.common.Actor;
import com.fasc.open.api.bean.common.ActorCorpMember;
import com.fasc.open.api.bean.common.OpenId;
import com.fasc.open.api.enums.corp.CorpIdentTypeEnum;
import com.fasc.open.api.v5_1.client.OpenApiClient;
import com.fasc.open.api.v5_1.client.SignTaskClient;
import com.fasc.open.api.v5_1.req.signtask.*;
import com.fasc.open.api.v5_1.res.signtask.CreateSignTaskRes;
import com.fasc.open.api.v5_1.res.signtask.GetSignTaskEditUrlRes;
import com.fasc.open.api.v5_1.res.signtask.OwnerDownloadUrlRes;
import com.fasc.open.api.v5_1.res.signtask.SignTaskActorGetUrlRes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.util.*;

/**
 * description：百川信法大大签署任务表 服务实现类
 * author：linchunpeng
 * date：2024/7/11
 */
@Slf4j
@Service
public class BfSignTaskService extends ServiceImpl<BfSignTaskMapper, BfSignTaskEntity> {

    @Autowired
    private FddConfig fddConfig;
    @Autowired
    private BfOrganizationService bfOrganizationService;
    @Autowired
    private FddApiUtilService fddApiUtilService;
    @Autowired
    private BfOrganizationMapper bfOrganizationMapper;
    @Autowired
    private IdGeneratorSnowflake snowflake;

    /**
     * 根据signTaskId查询签署任务
     */
    public BfSignTaskEntity findBySignTaskId(String signTaskId) {
        LambdaQueryChainWrapper<BfSignTaskEntity> lqw = this.lambdaQuery();
        lqw.eq(BfSignTaskEntity::getFddSignTaskId, signTaskId);
        List<BfSignTaskEntity> list = lqw.list();
        return CollectionUtil.isNotEmpty(list) ? list.get(0) : null;
    }

    /**
     * 发起签署任务，并返回法大大签署任务编辑链接
     */
    public GetFddUrlResponse signTaskCreate(SignTaskCreateRequest request) {
        try {
            GetFddUrlResponse response = new GetFddUrlResponse();
            Date now = new Date();

            AssertUtils.isTrue(Validator.isMobile(request.getUserMobile()), "签署方-对方手机号格式不正确");
            if (request.getTaskExpiresTime() != null) {
                AssertUtils.isTrue(request.getTaskExpiresTime().getTime() - now.getTime() > 10*60*1000, "任务截止时间不能小于10分钟");
            }
            if (request.getContractDueTime() != null && request.getTaskExpiresTime() != null) {
                AssertUtils.isTrue(DateUtil.getDay(request.getContractDueTime()) - DateUtil.getDay(request.getTaskExpiresTime()) > 0, "合同到期日期需要大于任务过期时间当天");
            }

            //获取开通的企业信息
            BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(request.getBcxOrganizationId());
            AssertUtils.isTrue(organizationEntity != null && organizationEntity.getAuthStatus() == 1, "企业未授权法大大");

            Map<String, Object> paramMap = new HashMap<>();
            BcxTenantUserResponse employeeUser = new BcxTenantUserResponse();
            if (request.getTaskType() == TaskTypeEnum.LABOR_CONTRACT.getCode()) {
                //劳动合同，。员工id必填
                AssertUtils.isTrue(StringUtils.isNotBlank(request.getBcxEmployeeId()), "职员id不能为空");
                paramMap.put("employeeId", request.getBcxEmployeeId());
                employeeUser = bfOrganizationMapper.selectTenantUserByEmployeeId(paramMap);
                AssertUtils.isNotNullObject(employeeUser, "员工不存在");
            } else {
                //其他合同，签署对方必填
                AssertUtils.isNotEmptyString(request.getUserName(), "签署方-对方不能为空");
            }
            paramMap.put("userId", organizationEntity.getBcxOrganizationAdminId());
            BcxTenantUserResponse adminUser = bfOrganizationMapper.selectTenantUserById(paramMap);
            AssertUtils.isNotNullObject(adminUser, "员工不存在");

            //创建签署任务记录
            BfSignTaskEntity bfSignTaskEntity = new BfSignTaskEntity();
            bfSignTaskEntity.setId(snowflake.snowflakeId());
            bfSignTaskEntity.setTaskType(request.getTaskType());
            bfSignTaskEntity.setTaskName(request.getTaskName());
            bfSignTaskEntity.setBcxOrganizationId(request.getBcxOrganizationId());
            if (request.getTaskType() == TaskTypeEnum.LABOR_CONTRACT.getCode()) {
                bfSignTaskEntity.setBcxEmployeeId(request.getBcxEmployeeId());
                bfSignTaskEntity.setUserName(employeeUser.getUserName());
            } else {
                bfSignTaskEntity.setUserName(request.getUserName());
            }
            bfSignTaskEntity.setUserMobile(request.getUserMobile());
            bfSignTaskEntity.setTaskExpiresTime(request.getTaskExpiresTime());
            bfSignTaskEntity.setContractStartTime(request.getContractStartTime());
            bfSignTaskEntity.setContractDueTime(request.getContractDueTime());
            bfSignTaskEntity.setTaskStatus(TaskStatusEnum.START.getCode());
            bfSignTaskEntity.setSyncStatus(0);
            bfSignTaskEntity.setRemark(request.getRemark());
            bfSignTaskEntity.setCreateTime(new Date());
            this.save(bfSignTaskEntity);

            //接口sdk初始化
            OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
            String accessToken = fddApiUtilService.getAccessToken(openApiClient);

            //创建签署任务
            SignTaskClient signTaskClient = new SignTaskClient(openApiClient);
            CreateSignTaskReq req = new CreateSignTaskReq();
            req.setAccessToken(accessToken);

            //发起方
            OpenId initiator = new OpenId();
            initiator.setIdType(CorpIdentTypeEnum.CORP.getCode());
            initiator.setOpenId(organizationEntity.getFddOpenCorpId());
            req.setInitiator(initiator);
            req.setInitiatorEntityId(organizationEntity.getFddEntityId());
            req.setInitiatorMemberId(organizationEntity.getFddAdminMemberId());

            //合同信息
            req.setSignTaskSubject(request.getTaskName());
            req.setSignDocType("contract");
            if (request.getTaskExpiresTime() != null) {
                req.setExpiresTime("" + request.getTaskExpiresTime().getTime());
            }
            if (request.getContractDueTime() != null) {
                req.setDueDate("" + request.getContractDueTime().getTime());
            }

            //签署双方信息
            List<AddActorsInfo> actors = new ArrayList<>();
            //企业
            AddActorsInfo org = new AddActorsInfo();
            Actor orgActor = new Actor();
            orgActor.setActorId(organizationEntity.getName());
            orgActor.setActorType(CorpIdentTypeEnum.CORP.getCode());
            orgActor.setActorName(organizationEntity.getName());
            orgActor.setPermissions(ListUtil.of("sign"));
            orgActor.setActorOpenId(organizationEntity.getFddOpenCorpId());
            List<ActorCorpMember> actorCorpMembers = new ArrayList<>();
            ActorCorpMember actorCorpMember = new ActorCorpMember();
            actorCorpMember.setMemberId(organizationEntity.getFddAdminMemberId());
            actorCorpMembers.add(actorCorpMember);
            orgActor.setActorCorpMembers(actorCorpMembers);
            if (Validator.isMobile(adminUser.getMobile())) {
                orgActor.setAccountName(adminUser.getMobile());
            }
            orgActor.setClientUserId(organizationEntity.getBcxOrganizationAdminId());
            org.setActor(orgActor);
            actors.add(org);
            //签署对方
            AddActorsInfo target = new AddActorsInfo();
            Actor targetActor = new Actor();
            if (request.getTaskType() == TaskTypeEnum.LABOR_CONTRACT.getCode()) {
                targetActor.setActorId(employeeUser.getUserName());
                targetActor.setActorName(employeeUser.getUserName());
            } else {
                targetActor.setActorId(request.getUserName());
                targetActor.setActorName(request.getUserName());
            }
            targetActor.setActorType("person");
            targetActor.setPermissions(ListUtil.of("sign"));
            targetActor.setAccountName(request.getUserMobile());
            target.setActor(targetActor);
            actors.add(target);
            req.setActors(actors);

            BaseRes<CreateSignTaskRes> res = signTaskClient.create(req);
            log.info("发起签署任务，法大大返回值：{}", JSONObject.toJSONString(res));
            AssertUtils.isTrue(res.getCode().equals("100000"), res.getMsg());
            log.info("任务发起成功，修改任务记录");
            BfSignTaskEntity bfSignTask = this.getById(bfSignTaskEntity.getId());
            bfSignTask.setFddSignTaskId(res.getData().getSignTaskId());
            this.updateById(bfSignTask);
            log.info("获取签署任务编辑链接");
            GetSignTaskUrlReq getSignTaskUrlReq = new GetSignTaskUrlReq();
            getSignTaskUrlReq.setAccessToken(accessToken);
            getSignTaskUrlReq.setSignTaskId(bfSignTask.getFddSignTaskId());
            getSignTaskUrlReq.setRedirectUrl("https://v5qy.te.baibaodun.com.cn/static/portal/vue/index.html?v=1724732498838#/");
            BaseRes<GetSignTaskEditUrlRes> signTaskEditUrl = signTaskClient.getSignTaskEditUrl(getSignTaskUrlReq);
            AssertUtils.isTrue(signTaskEditUrl.getCode().equals("100000"), res.getMsg());
            response.setUrl(signTaskEditUrl.getData().getSignTaskEditUrl());
            return response;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("发起签署任务，并返回法大大签署任务编辑链接异常，{}", e.getMessage(), e);
            throw AssertException.exception(e.getMessage());
        }
    }

    /**
     * 获取签署任务编辑链接
     */
    public GetFddUrlResponse signTaskGetEditUrl(SignTaskActorGetUrlRequest request) {
        try {
            GetFddUrlResponse response = new GetFddUrlResponse();
            //获取开通的企业信息
            BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(request.getBcxOrganizationId());
            AssertUtils.isTrue(organizationEntity != null && organizationEntity.getAuthStatus() == 1, "企业未授权法大大");
            BfSignTaskEntity signTaskEntity = this.getById(request.getBfSignTaskId());
            AssertUtils.isNotNullObject(signTaskEntity, "签署任务不存在");

            //接口sdk初始化
            OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
            String accessToken = fddApiUtilService.getAccessToken(openApiClient);

            SignTaskClient signTaskClient = new SignTaskClient(openApiClient);
            GetSignTaskUrlReq req = new GetSignTaskUrlReq();
            req.setAccessToken(accessToken);
            req.setSignTaskId(signTaskEntity.getFddSignTaskId());
            req.setRedirectUrl("https://v5qy.te.baibaodun.com.cn/static/portal/vue/index.html?v=1724732498838#/");
            //不可编辑项
            req.setNonEditableInfo(ListUtil.of("basicInfo", "actors"));
            BaseRes<GetSignTaskEditUrlRes> res = signTaskClient.getSignTaskEditUrl(req);
            AssertUtils.isTrue(res.getCode().equals("100000"), res.getMsg());
            response.setUrl(res.getData().getSignTaskEditUrl());
            return response;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取签署任务编辑链接异常，{}", e.getMessage(), e);
            throw AssertException.exception(e.getMessage());
        }
    }

    /**
     * 获取企业签署链接
     */
    public GetFddUrlResponse signTaskActorGetUrl(SignTaskActorGetUrlRequest request) {
        try {
            GetFddUrlResponse response = new GetFddUrlResponse();
            //获取开通的企业信息
            BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(request.getBcxOrganizationId());
            AssertUtils.isTrue(organizationEntity != null && organizationEntity.getAuthStatus() == 1, "企业未授权法大大");
            BfSignTaskEntity signTaskEntity = this.getById(request.getBfSignTaskId());
            AssertUtils.isNotNullObject(signTaskEntity, "签署任务不存在");

            //接口sdk初始化
            OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
            String accessToken = fddApiUtilService.getAccessToken(openApiClient);

            SignTaskClient signTaskClient = new SignTaskClient(openApiClient);
            SignTaskActorGetUrlReq req = new SignTaskActorGetUrlReq();
            req.setAccessToken(accessToken);

            //发起方
            req.setSignTaskId(signTaskEntity.getFddSignTaskId());
            req.setActorId(organizationEntity.getName());
            req.setClientUserId(organizationEntity.getBcxOrganizationAdminId());
            req.setRedirectUrl("https://v5qy.te.baibaodun.com.cn/static/portal/vue/index.html?v=1724732498838#/");

            BaseRes<SignTaskActorGetUrlRes> res = signTaskClient.signTaskActorGetUrl(req);
            log.info("获取企业签署链接，法大大返回值：{}", JSONObject.toJSONString(res));
            AssertUtils.isTrue(res.getCode().equals("100000"), res.getMsg());
            response.setUrl(res.getData().getActorSignTaskEmbedUrl());
            return response;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取企业签署链接异常，{}", e.getMessage(), e);
            throw AssertException.exception(e.getMessage());
        }
    }


    /**
     * 下载合同附件
     */
    public void downloadTaskFile(Long bfSignTaskId) {
        try {
            //获取开通的企业信息
            BfSignTaskEntity signTaskEntity = this.getById(bfSignTaskId);
            if (signTaskEntity != null) {
                BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(signTaskEntity.getBcxOrganizationId());
                //接口sdk初始化
                OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
                String accessToken = fddApiUtilService.getAccessToken(openApiClient);

                //获取签署文档下载地址
                SignTaskClient signTaskClient = new SignTaskClient(openApiClient);
                GetOwnerDownloadUrlReq req = new GetOwnerDownloadUrlReq();
                req.setAccessToken(accessToken);

                //发起方
                req.setSignTaskId(signTaskEntity.getFddSignTaskId());
                OpenId ownerId = new OpenId();
                ownerId.setIdType("corp");
                ownerId.setOpenId(organizationEntity.getFddOpenCorpId());
                req.setOwnerId(ownerId);

                BaseRes<OwnerDownloadUrlRes> res = signTaskClient.getOwnerDownloadUrl(req);
                log.info("获取签署文档下载地址，法大大返回值：{}", JSONObject.toJSONString(res));
                AssertUtils.isTrue(res.getCode().equals("100000"), res.getMsg());
                String downloadUrl = res.getData().getDownloadUrl();
                String uuid = UUID.randomUUID().toString();
                String fileSavePath = FileUtils.downloadByUrl(downloadUrl, fddConfig.getFileSavePath().concat(uuid));
                if ("fail".equals(fileSavePath)) {
                    signTaskEntity.setTaskFileObsUrl("fail");
                } else {
                    String obsPath = fileSavePath.substring(fileSavePath.indexOf("fddfile/"));
                    ObsUtil.uploadFile(new File(fileSavePath), obsPath);
                    String obsUrl = "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/";
                    signTaskEntity.setTaskFileObsUrl(obsUrl.concat(obsPath));
                }
                signTaskEntity.setUpdateTime(new Date());
                this.updateById(signTaskEntity);
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("下载合同附件异常，{}", e.getMessage(), e);
        }
    }

    /**
     * 获取签署文档下载地址
     */
    public GetFddUrlResponse getOwnerDownloadUrl(SignTaskActorGetUrlRequest request) {
        try {
            GetFddUrlResponse response = new GetFddUrlResponse();
            //获取开通的企业信息
            BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(request.getBcxOrganizationId());
            AssertUtils.isTrue(organizationEntity != null && organizationEntity.getAuthStatus() == 1, "企业未授权法大大");
            BfSignTaskEntity signTaskEntity = this.getById(request.getBfSignTaskId());
            AssertUtils.isNotNullObject(signTaskEntity, "签署任务不存在");

            //接口sdk初始化
            OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
            String accessToken = fddApiUtilService.getAccessToken(openApiClient);

            //获取签署文档下载地址
            SignTaskClient signTaskClient = new SignTaskClient(openApiClient);
            GetOwnerDownloadUrlReq req = new GetOwnerDownloadUrlReq();
            req.setAccessToken(accessToken);

            //发起方
            req.setSignTaskId(signTaskEntity.getFddSignTaskId());
            OpenId ownerId = new OpenId();
            ownerId.setIdType("corp");
            ownerId.setOpenId(organizationEntity.getFddOpenCorpId());
            req.setOwnerId(ownerId);

            BaseRes<OwnerDownloadUrlRes> res = signTaskClient.getOwnerDownloadUrl(req);
            log.info("获取签署文档下载地址，法大大返回值：{}", JSONObject.toJSONString(res));
            AssertUtils.isTrue(res.getCode().equals("100000"), res.getMsg());
            response.setUrl(res.getData().getDownloadUrl());
            return response;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取签署文档下载地址异常，{}", e.getMessage(), e);
            throw AssertException.exception(e.getMessage());
        }
    }
}