package com.bcxin.fdd.service;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.fdd.common.exception.AssertException;
import com.bcxin.fdd.common.utils.AssertUtils;
import com.bcxin.fdd.common.utils.IdGeneratorSnowflake;
import com.bcxin.fdd.dao.mapper.BfOrganizationMapper;
import com.bcxin.fdd.dtos.request.OrganizationOpenRequest;
import com.bcxin.fdd.dtos.request.SharedMessageRequest;
import com.bcxin.fdd.dtos.response.BcxOrganizationResponse;
import com.bcxin.fdd.dtos.response.BcxTenantUserResponse;
import com.bcxin.fdd.dtos.response.GetFddUrlResponse;
import com.bcxin.fdd.entity.BfOrganizationEntity;
import com.fasc.open.api.bean.base.BaseRes;
import com.fasc.open.api.enums.corp.CorpAuthScopeEnum;
import com.fasc.open.api.enums.corp.CorpIdentTypeEnum;
import com.fasc.open.api.enums.user.UserIdentTypeEnum;
import com.fasc.open.api.v5_1.client.CorpClient;
import com.fasc.open.api.v5_1.client.OpenApiClient;
import com.fasc.open.api.v5_1.client.OrgClient;
import com.fasc.open.api.v5_1.req.corp.CorpIdentInfoReq;
import com.fasc.open.api.v5_1.req.corp.GetCorpAuthResourceUrlReq;
import com.fasc.open.api.v5_1.req.corp.GetCorpReq;
import com.fasc.open.api.v5_1.req.corp.OprIdentInfoReq;
import com.fasc.open.api.v5_1.req.org.GetEntityListReq;
import com.fasc.open.api.v5_1.req.org.GetMemberListFilter;
import com.fasc.open.api.v5_1.req.org.GetMemberListReq;
import com.fasc.open.api.v5_1.res.common.ECorpAuthUrlRes;
import com.fasc.open.api.v5_1.res.corp.CorpRes;
import com.fasc.open.api.v5_1.res.org.EmployeeInfo;
import com.fasc.open.api.v5_1.res.org.GetEntityListRes;
import com.fasc.open.api.v5_1.res.org.GetMemberListRes;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * description：百川信法大大组织表 服务实现类
 * author：linchunpeng
 * date：2024/7/11
 */
@Slf4j
@Service
public class BfOrganizationService extends ServiceImpl<BfOrganizationMapper, BfOrganizationEntity> {

    @Autowired
    private IdGeneratorSnowflake snowflake;
    @Autowired
    private BfOrganizationMapper bfOrganizationMapper;
    @Autowired
    private FddApiUtilService fddApiUtilService;

    @Value("${org-auth-config.notice-mobile}")
    private String noticeMobile;

    /**
     * 根据百川信组织id获取企业信息
     */
    public BfOrganizationEntity getByBcxOrganizationId(String bcxOrganizationId) {
        LambdaQueryChainWrapper<BfOrganizationEntity> lqw = this.lambdaQuery();
        lqw.eq(BfOrganizationEntity::getBcxOrganizationId, bcxOrganizationId);
        List<BfOrganizationEntity> list = lqw.list();
        return CollectionUtil.isNotEmpty(list) ? list.get(0) : null;
    }

    /**
     * 根据百川信组织id获取企业信息
     */
    @Transactional
    public BfOrganizationEntity getByBcxOrganizationId(OrganizationOpenRequest request) {
        BfOrganizationEntity organizationEntity = this.getByBcxOrganizationId(request.getBcxOrganizationId());
        if (organizationEntity == null) {
            //不存在则新增
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("bcxOrganizationId", request.getBcxOrganizationId());
            BcxOrganizationResponse organizationResponse = bfOrganizationMapper.selectOrganizationById(paramMap);
            AssertUtils.isNotNullObject(organizationResponse, "该企业id不存在");

            BfOrganizationEntity entity = new BfOrganizationEntity();
            entity.setId(snowflake.snowflakeId());
            entity.setBcxOrganizationId(organizationResponse.getId());
            entity.setBcxOrganizationAdminId(organizationResponse.getAdminId());
            entity.setName(organizationResponse.getOrgName());
            entity.setCode(organizationResponse.getOrgCode());
            entity.setLegalPersonName(organizationResponse.getLegalPersonName());
            entity.setAuthStatus(0);
            entity.setStatus(1);
            entity.setRemark(request.getRemark());
            entity.setCreateTime(new Date());
            entity.setUpdateTime(new Date());
            this.saveOrUpdate(entity);
            return entity;
        } else {
            return organizationEntity;
        }
    }

    /**
     * 获取企业授权链接
     */
    public GetFddUrlResponse getAuthUrl(OrganizationOpenRequest request) {
        try {
            GetFddUrlResponse response = new GetFddUrlResponse();
            //获取开通的企业信息
            BfOrganizationEntity organizationEntity = this.getByBcxOrganizationId(request);
            AssertUtils.isTrue(organizationEntity.getAuthStatus() == 0, "该企业已经授权");
            //接口sdk初始化
            OpenApiClient openApiClient = fddApiUtilService.getOpenApiClient();
            String accessToken = fddApiUtilService.getAccessToken(openApiClient);
            CorpClient corpClient = new CorpClient(openApiClient);
            // 获取企业用户授权链接
            GetCorpAuthResourceUrlReq req = new GetCorpAuthResourceUrlReq();
            req.setAccessToken(accessToken);

            //企业在应用中的唯一标识
            req.setClientCorpId(organizationEntity.getBcxOrganizationId());
            //管理用户id
            req.setClientUserId(organizationEntity.getBcxOrganizationAdminId());

            //企业基本信息
            CorpIdentInfoReq corpIdentInfo = new CorpIdentInfoReq();
            corpIdentInfo.setCorpName(organizationEntity.getName());
            corpIdentInfo.setCorpIdentType(CorpIdentTypeEnum.CORP.getCode());
            corpIdentInfo.setCorpIdentNo(organizationEntity.getCode());
            corpIdentInfo.setLegalRepName(organizationEntity.getLegalPersonName());
            req.setCorpIdentInfo(corpIdentInfo);

            //经办人基本信息
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("userId", organizationEntity.getBcxOrganizationAdminId());
            BcxTenantUserResponse tenantUserResponse = bfOrganizationMapper.selectTenantUserById(paramMap);
            OprIdentInfoReq oprIdentInfo = new OprIdentInfoReq();
            oprIdentInfo.setUserName(tenantUserResponse.getUserName());
            oprIdentInfo.setUserIdentType(UserIdentTypeEnum.ID_CARD.getCode());
            oprIdentInfo.setUserIdentNo(tenantUserResponse.getIdCardNo());
            oprIdentInfo.setMobile(tenantUserResponse.getMobile());
            req.setOprIdentInfo(oprIdentInfo);

            //（可选）法大大平台上企业主体的名称
            req.setCorpName(organizationEntity.getName());
            //（可选）企业组织类型CorpIdentTypeEnum
            req.setCorpIdentType(CorpIdentTypeEnum.CORP.getCode());
            //（可选）企业统一社会信用代码或各种类型组织的唯一代码
            req.setCorpIdentNo(organizationEntity.getCode());
            //（可选）是否需要匹配企业身份信息一致，缺省为false。
            req.setCorpIdentInfoMatch(false);
            //（可选）企业授权范围列表CorpAuthScopeEnum
            req.setAuthScopes(Arrays.asList(CorpAuthScopeEnum.IDENT_INFO.getCode(),
                    CorpAuthScopeEnum.SIGN_TASK_INIT.getCode(),
                    CorpAuthScopeEnum.SIGN_TASK_INFO.getCode(),
                    CorpAuthScopeEnum.SIGN_TASK_FILE.getCode(),
                    CorpAuthScopeEnum.SEAL_INFO.getCode(),
                    CorpAuthScopeEnum.ORGANIZATION.getCode(),
                    CorpAuthScopeEnum.TEMPLATE.getCode(),
                    "contract_info", "billaccount_info", "smartform"));

            req.setRedirectUrl("https://v5qy.te.baibaodun.com.cn/static/portal/vue/index.html?v=1724732498838#/");

            BaseRes<ECorpAuthUrlRes> res = corpClient.getCorpAuthUrl(req);
            log.info("获取企业授权链接，法大大返回值：{}", JSONObject.toJSONString(res));
            if (res.getCode().equals("210002")) {
                //已经授权过了，要获取授权信息
                GetCorpReq getCorpReq = new GetCorpReq();
                getCorpReq.setAccessToken(accessToken);
                getCorpReq.setCorpIdentNo(organizationEntity.getCode());
                BaseRes<CorpRes> corpResBaseRes = corpClient.get(getCorpReq);
                log.info("获取授权信息，法大大返回值：{}", JSONObject.toJSONString(corpResBaseRes));
                organizationEntity.setFddClientCorpId(corpResBaseRes.getData().getClientCorpId());
                organizationEntity.setFddOpenCorpId(corpResBaseRes.getData().getOpenCorpId());

                OrgClient orgClient = new OrgClient(openApiClient);
                GetEntityListReq getEntityListReq = new GetEntityListReq();
                getEntityListReq.setAccessToken(accessToken);
                getEntityListReq.setOpenCorpId(organizationEntity.getFddOpenCorpId());
                BaseRes<List<GetEntityListRes>> corpEntityList = orgClient.getCorpEntityList(getEntityListReq);
                if (CollectionUtil.isNotEmpty(corpEntityList.getData())) {
                    log.info("获取企业主体信息，法大大返回值：{}", JSONObject.toJSONString(corpEntityList));
                    Optional<GetEntityListRes> primary = corpEntityList.getData().stream().filter(getEntityListRes -> getEntityListRes.getEntityType().equals("primary")).findFirst();
                    primary.ifPresent(getEntityListRes -> organizationEntity.setFddEntityId(getEntityListRes.getEntityId()));
                }

                GetMemberListReq getMemberListReq = new GetMemberListReq();
                getMemberListReq.setAccessToken(accessToken);
                getMemberListReq.setOpenCorpId(organizationEntity.getFddOpenCorpId());
                GetMemberListFilter getMemberListFilter = new GetMemberListFilter();
                getMemberListFilter.setRoleType("super_admin");
                getMemberListReq.setListFilter(getMemberListFilter);
                BaseRes<GetMemberListRes> memberList = orgClient.getMemberList(getMemberListReq);
                if (CollectionUtil.isNotEmpty(memberList.getData().getEmployeeInfos())) {
                    log.info("获取企业成员信息，法大大返回值：{}", JSONObject.toJSONString(memberList));
                    EmployeeInfo employeeInfo = memberList.getData().getEmployeeInfos().get(0);
                    organizationEntity.setFddAdminUserId(organizationEntity.getBcxOrganizationAdminId());
                    organizationEntity.setFddAdminMemberId(employeeInfo.getMemberId().toString());
                }
                organizationEntity.setAuthStatus(1);
                organizationEntity.setUpdateTime(new Date());
                this.updateById(organizationEntity);
                response.setMsg("该企业已经授权");

                this.insertOrgAuthNotice(organizationEntity);
                return response;
            }
            response.setUrl(res.getData().getAuthUrl());
            return response;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取企业授权链接异常，{}", e.getMessage(), e);
            throw AssertException.exception(e.getMessage());
        }
    }

    /**
     * 企业授权成功，短信通知运营人员授权份额
     */
    @Async
    @Transactional
    public void insertOrgAuthNotice(BfOrganizationEntity organizationEntity) {
        try {
            String title = "【百保盾】";
            String content = "【百保盾】企业：" + organizationEntity.getName() + "，已完成法大大企业认证，并授权给【百保盾】平台。";
            log.info("企业授权成功，短信通知运营人员授权份额，短信内容：{}", content);
            bfOrganizationMapper.insertOrgAuthNotice(new SharedMessageRequest(noticeMobile, title, content));
        } catch (Exception e) {
            log.info("企业授权成功，短信通知运营人员授权份额出错，{}", e.getMessage(), e);
        }
    }
}