package com.bcxin.fdd.common.utils;

import com.bcxin.fdd.common.exception.AssertException;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * 断言帮助类，主要用户需要抛异常的参数校验。 对于必须校验通过的参数，建议使用该类处理。
 */
public abstract class AssertUtils {

    /**
     * 默认的消息，没有指定消息的断言都使用这个
     */
    private static final String DEFAULT_MESSAGE = "数据校验错误";

    private AssertUtils() {
    }

    //region 字符串断言

    /**
     * 判断是非空的字符串，如果字符串为空，那么抛出异常。
     *
     * @param string
     * @throws AssertException
     */
    public static void isNotEmptyString(String string) {
        isTrue(StringUtils.isNotEmpty(string), DEFAULT_MESSAGE);
    }

    /**
     * 判断表达式是否为真,带自定义消息。
     * 如果不是true则抛出异常。
     *
     * @param flag
     * @param message
     * @throws AssertException
     */
    public static void isTrue(boolean flag, String message) {
        if (!flag) {
            throw AssertException.exception(message);
        }
    }

    /**
     * 判断是非空的字符串，如果字符串为空，那么抛出异常。带自定义消息。
     *
     * @param string
     * @param message
     * @throws AssertException
     */
    public static void isNotEmptyString(String string, String message) {
        isTrue(StringUtils.isNotEmpty(string), message);
    }

    /**
     * 判断是非空的字符串,带格式化消息
     *
     * @param string
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isNotEmptyString(String string, String messageFormat, Object... args) {
        isTrue(StringUtils.isNotEmpty(string), messageFormat, args);
    }

    /**
     * 判断表达式是否为true,带格式化消息
     *
     * @param flag
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isTrue(boolean flag, String messageFormat, Object... args) {
        if (!flag) {
            throw AssertException.exception(messageFormat, args);
        }
    }

    /**
     * 判断是空的字符串，如果字符串为空，那么抛出异常。
     *
     * @param string
     * @throws AssertException
     */
    public static void isEmptyString(String string) {
        isTrue(StringUtils.isBlank(string), DEFAULT_MESSAGE);
    }
    //endregion


    // region 真假判断

    /**
     * 判断是空的字符串，如果字符串为空，那么抛出异常。带自定义消息。
     *
     * @param string
     * @param message
     * @throws AssertException
     */
    public static void isEmptyString(String string, String message) {
        isTrue(StringUtils.isBlank(string), message);
    }

    /**
     * 判断是空的字符串,带格式化消息
     *
     * @param string
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isEmptyString(String string, String messageFormat, Object... args) {
        isTrue(StringUtils.isBlank(string), messageFormat, args);
    }

    /**
     * 判断表达式是否为true，不带消息
     *
     * @param flag
     * @throws AssertException
     */
    public static void isTrue(boolean flag) {
        isTrue(flag, DEFAULT_MESSAGE);
    }

    /**
     * 判断表达式是否为false，不带消息
     *
     * @param flag
     * @throws AssertException
     */
    public static void isFalse(boolean flag) {
        isFalse(flag, DEFAULT_MESSAGE);
    }

    /**
     * 判断表达式是否为false,带消息
     *
     * @param flag
     * @param message
     * @throws AssertException
     */
    public static void isFalse(boolean flag, String message) {
        if (flag) {
            throw AssertException.exception(message);
        }
    }

    /**
     * 判断表达式是否为false,带格式化消息
     *
     * @param flag
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isFalse(boolean flag, String messageFormat, Object... args) {
        if (flag) {
            throw AssertException.exception(messageFormat, args);
        }
    }
    // endregion


    // region List判断

    /**
     * 判断列表是否为非空列表，如果列表是空抛出异常
     *
     * @param list
     * @throws AssertException
     */
    public static void isNotEmptyList(List list) {
        isTrue(list != null && list.size() > 0, DEFAULT_MESSAGE);
    }

    /**
     * 判断列表是否为非空列表，如果列表是空抛出异常
     *
     * @param list
     * @param message
     * @throws AssertException
     */
    public static void isNotEmptyList(List list, String message) {
        isTrue(list != null && list.size() > 0, message);
    }

    /**
     * 判断列表是否为非空列表，如果列表是空抛出异常
     *
     * @param list
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isNotEmptyList(List<?> list, String messageFormat, Object... args) {
        isTrue(list != null && list.size() > 0, messageFormat, args);
    }
    // endregion

    // region Map判断

    /**
     * 判断Map是否不为空，如果为空抛出异常
     *
     * @param map
     * @throws AssertException
     */
    public static void isNotEmptyMap(Map<?, ?> map) {
        isTrue(map != null && map.size() > 0, DEFAULT_MESSAGE);
    }

    /**
     * 判断Map是否不为空，如果为空抛出异常
     *
     * @param map
     * @param message
     * @throws AssertException
     */
    public static void isNotEmptyMap(Map<?, ?> map, String message) {
        isTrue(map != null && map.size() > 0, message);
    }

    /**
     * 判断Map是否不为空，如果为空抛出异常
     *
     * @param map
     * @param messageFormat
     * @param args
     * @throws AssertException
     */
    public static void isNotEmptyMap(Map<?, ?> map, String messageFormat, Object... args) {
        isTrue(map != null && map.size() > 0, messageFormat, args);
    }

    // endregion

    // region 对象判断

    /**
     * 判断对象是否不为空，如果为空那么抛出异常
     *
     * @param object
     * @throws AssertException
     */
    public static void isNotNullObject(Object object) {
        isTrue(object != null, DEFAULT_MESSAGE);
    }

    /**
     * 判断对象是否不为空，如果为空那么抛出异常
     *
     * @param object
     * @param message
     * @throws AssertException
     */
    public static void isNotNullObject(Object object, String message) {
        isTrue(object != null, message);
    }

    /**
     * 判断对象是否不为空，如果为空那么抛出异常
     *
     * @param object
     * @param messageFormat
     * @param args
     */
    public static void isNotNullObject(Object object, String messageFormat, Object... args) {
        isTrue(object != null, messageFormat, args);
    }

}
