package com.bcxin.ars.timer.dataSync;

import com.abcxin.smart.validator.annotation.DataSyncAnnotation;
import com.abcxin.smart.validator.annotation.JobLogAnnotation;
import com.bcxin.ars.dto.DataSynchronizationSearchDto;
import com.bcxin.ars.model.Config;
import com.bcxin.ars.service.util.ConfigUtils;
import com.bcxin.ars.service.util.DataSyncUtil;
import com.bcxin.ars.util.Constants;
import com.bcxin.ars.util.DateUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.File;
import java.util.Date;
import java.util.List;

/**
 * 内外网数据同步
 * 需要同步的表在service实现类加上DataSyncAnnotation注解
 * service实现类中
 *  读取同步数据文件实现saveOrUpdateForDS或updateFlagForOutToIn
 *  生成同步数据文件实现searchForDataSynchronization
 * @author linqinglin
 * @date 2019/04/03 14:15
 */
@Service
public class DataSyncDSService {

    private Logger logger = LoggerFactory.getLogger(DataSyncDSService.class);

    @Autowired
    private ConfigUtils configUtils;

    @Autowired
    private DataSyncUtil dataSyncUtil;

    @Value("${ftp-folder}")
    private String ftpFolder;

    private boolean lock = false;

    /***
     *  增量数据
     */
    @JobLogAnnotation(getJobClazz =DataSyncDSService.class)
    public void run() {
        if(lock){
            return;
        }
        lock = true;

        try {
            if(Constants.DATASYNC_SWITCH_1.equals(configUtils.getValueByKey(Constants.DATASYNC_SWITCH))){
                return;
            }
            //获取配置是否要同步
             Config datasynchronizationconfig = configUtils.getConfigByKey(Constants.DATASYNCHRONIZATION);
            //同步
            if (datasynchronizationconfig != null
                    && Constants.DATASYNCHRONIZATION_YES.equals(datasynchronizationconfig.getValue())
                    && Constants.TIMEFLAG_VALUE.equals(configUtils.timeFlag)) {
                if (configUtils.isIntranet()) {
                    //内网读取同步文件
                    policeInReadFile();
                }else{
                    DataSynchronizationSearchDto searchDto = new DataSynchronizationSearchDto();
                    searchDto.setPaging(false);
                    /***
                     * 最后同步日期
                     */
                    Config lastDateTimeConfig = configUtils.getConfigByKey(Constants.LASTDATETIME);
                    //配置为空或者等于当天日期
                    if(lastDateTimeConfig == null || DateUtil.convertDateToString(new Date(), DateUtil.FORMAT2).equals(lastDateTimeConfig.getValue())){
                        searchDto.setStartDate(DateUtil.getYesterdayTime());
                    }else{
                        //设置为最后同步日期
                        searchDto.setStartDate(lastDateTimeConfig.getValue());
                    }
                    searchDto.setEndDate(DateUtil.getTomorrowTime());
                    //外网生成同步文件
                    policeOutCreateFile(searchDto);
                    /**
                     * 更新"外到内最后同步日期" start
                     */
                    Config updateConfig = new Config();
                    updateConfig.setKey(Constants.LASTDATETIME);
                    updateConfig.setValue(DateUtil.getDateStrAfterMinute(-5));
                    updateConfig.setUpdateTime(new Date());
                    configUtils.updateForKey(updateConfig);
                    /**
                     * 更新"外到内最后同步日期" end
                     */
                }
            }
        }finally {
            lock = false;
        }
    }
    private void policeInReadFile() {
        Date now = new Date();
        /***
         * 最后同步日期
         */
        Config lastDateConfig = configUtils.getConfigByKey(Constants.LASTDATE);
        if (lastDateConfig != null) {
            //读取的时间为最后更新的时间
            String lastDate = lastDateConfig.getValue();
            List<Date> dates = DateUtil.getBetweenDates(DateUtil.convertStringToDate(lastDate), now);
            //加入当前日期
            dates.add(now);
            for (Date date : dates) {
                String dirFilePath = ftpFolder + DateUtil.convertDateToString(date, DateUtil.FORMAT2);
                File dirFile = new File(dirFilePath);
                if (!dirFile.exists()) {
                    break;
                }
                File[] files = dirFile.listFiles();
                //读取是否有错误
                boolean errorFlag = false;
                for (int i = 0; i < files.length; i++) {
                    File file = files[i];
                    if(file.getName().endsWith(Constants.ZIP) && file.getName().startsWith(configUtils.getDataSyncStartFileName(DataSyncAnnotation.class))){
                        try {
                            dataSyncUtil.readDSFile(DataSyncAnnotation.class,file);
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                            errorFlag = true;
                        }
                    }
                }

                //如果文件没有读取错误
                if (!errorFlag) {
                    Config updateConfig = new Config();
                    updateConfig.setKey(Constants.LASTDATE);
                    updateConfig.setValue(DateUtil.convertDateToString(date, DateUtil.FORMAT2));
                    configUtils.updateForKey(updateConfig);
                }

            }
        }
    }

   private void policeOutCreateFile(DataSynchronizationSearchDto searchDto) {
        dataSyncUtil.createDSFile(DataSyncAnnotation.class,searchDto,ftpFolder + DateUtil.getCurrentDate()+File.separator);
    }

}
