package com.bcxin.ars.timer.company;

import com.bcxin.ars.dao.sb.InvestorDao;
import com.bcxin.ars.dto.AjaxResult;
import com.bcxin.ars.dto.SAASMessageDto;
import com.bcxin.ars.dto.sb.InvestorDto;
import com.bcxin.ars.model.sys.JobRunLog;
import com.bcxin.ars.service.sys.JobRunLogService;
import com.bcxin.ars.service.util.ConfigUtils;
import com.bcxin.ars.service.util.SAASInterfaceUtil;
import com.bcxin.ars.util.Constants;
import com.bcxin.ars.util.text.StringUtils;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 股东信息不完整通知
 *
 * @author zhangyutong
 */
@Service
public class InvestorInfoIncompleteNotifyService {

    private static final Logger logger = LoggerFactory.getLogger(InvestorInfoIncompleteNotifyService.class);

    @Resource
    private JobRunLogService jobRunLogService;

    @Resource
    private SAASInterfaceUtil saasInterfaceUtil;

    @Resource
    private ConfigUtils configUtils;

    @Resource
    private InvestorDao investorDao;

    public void run() {
        // 判断是否是 北京外网环境
        if(!Constants.TIMEFLAG_VALUE.equals(configUtils.timeFlag)
                || !Constants.BEIJING.equals(configUtils.getCurrentNative())
                || configUtils.isIntranet()){
            return;
        }
        // 记录日志
        JobRunLog log = getJobRunLog();
        long startTime = System.currentTimeMillis();
        try {
            // 处理数据
            handleInfoIncomplete();
        } catch (RuntimeException e) {
            logger.error("股东信息站内信提醒异常:{}",e.getMessage());
            e.printStackTrace();
        } finally {
            long endTime = System.currentTimeMillis();
            log.setRunTimeLength((endTime - startTime) + "ms");
            log.setUpdateTime(new Date());
            // 保存日志
            jobRunLogService.insert(log);
        }
    }

    /**
     * 处理的实际方法
     */
    public void handleInfoIncomplete() {
        // 获取所有股东
        List<InvestorDto> investors = investorDao.listWithComId();
        // comId -> name1,name2, 因为一个公司有多个股东
        Map<String, StringBuilder> incompleteInfoMap = Maps.newHashMap();

        for (InvestorDto investor : investors) {
            // 如果是机构股东 则跳过
            if (Constants.INVESTOR_TYPE_ORG.equals(investor.getInvestorsType())) {
                continue;
            }
            // 判断字段信息是否完整
            boolean isIncomplete = isInfoIncomplete(investor);
            if (!isIncomplete) {
                // 不完整, 输出下, 存储起来
                String comId = investor.getComId();
                // 如果没有则先创建再追加
                StringBuilder stringBuilder = incompleteInfoMap.get(comId);
                if (stringBuilder == null) {
                    stringBuilder = new StringBuilder();
                }
                // 追加名字
                stringBuilder.append(investor.getName()).append("，");
                incompleteInfoMap.put(comId,stringBuilder);
            }
        }
        logger.info("需要发送站内信的公司及股东===>{}", incompleteInfoMap);
        if (incompleteInfoMap.isEmpty()) {
            return;
        }
        // 发送信息
        incompleteInfoMap.forEach((key, val) -> send(key, Constants.MESSAGE_COMPANY_INVESTOR_INFO_INCOMPLETE_TITLE, Constants.MESSAGE_COMPANY_INVESTOR_INFO_INCOMPLETE_CONTENT.replace("{investors}", val)));
    }


    /**
     * 反射获取所有参数, 一个为null则return false
     *
     * @param object 参数
     * @return 是否参数不完整
     */
    public boolean isInfoIncomplete(Object object) {
        // 查询出对象所有的属性
        Field[] fields = object.getClass().getDeclaredFields();
        for (Field field : fields) {
            // 不检查 直接取值
            field.setAccessible(true);
            try {
                String value = (String) field.get(object);
                if (StringUtils.isBlank(value)) {
                    // 当有任何一个参数为空的时候则跳出判断直接查询
                    return false;
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 日志
     *
     * @return 日志对象
     */
    public JobRunLog getJobRunLog() {
        // 设置日志记录
        JobRunLog log = new JobRunLog();
        log.setCreateTime(new Date());
        log.setJobName(this.getClass().getName());
        log.setActive(true);
        log.setUpdateBy("jobSystem");
        log.setRunTime(new Date());
        return log;
    }


    /**
     * 发送消息的方法,调用对方接口发站内信
     *
     * @param companyIds 公司id集合
     * @param title      消息头
     * @param content    消息体
     * @return 返回结果
     */
    public AjaxResult send(String companyIds,
                           String title,
                           String content) {
        SAASMessageDto saasMessageDto = new SAASMessageDto();
        saasMessageDto.setComIds(companyIds);
        saasMessageDto.setTitle(title);
        saasMessageDto.setContent(content);
        saasMessageDto.setPoliceRegionId(configUtils.getCurrentProvince());
        saasMessageDto.setMessageType(Constants.MESSAGETYPE_050303);
        return saasInterfaceUtil.sendSaasMsg(saasMessageDto);
    }

}
