package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.SystemConstant;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.ConfirmCheckStatusException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.Infrastructures.utils.JwtUtil;
import com.bcxin.Infrastructures.utils.StringUtils;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.ToCWechatCgiBinSignInRequest;
import com.bcxin.api.interfaces.identities.requests.TocUserAssignIdentityRequest;
import com.bcxin.api.interfaces.identities.responses.SignInResponse;
import com.bcxin.api.interfaces.identities.responses.ToCWechatCgiBinSignInResponse;
import com.bcxin.rest.web.apis.requests.TocUserConfirmCheckStatusRequest;
import com.bcxin.api.interfaces.identities.responses.TocUserConfirmCheckStatusResponse;
import com.bcxin.api.interfaces.identities.valueTypes.WechatCgiBinSubject;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.TocWechatCgiConfirmCheckStatusRequest;
import com.bcxin.api.interfaces.tenants.responses.TocWechatCgiConfirmCheckStatusResponse;
import com.bcxin.rest.web.apis.utils.IPUtils;
import com.bcxin.rest.web.apis.utils.ServletRequestUtil;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

@RestController
@RequestMapping("/wechat/cgi-bin")
public class WechatController extends ControllerAbstract {
    private final Logger logger = LoggerFactory.getLogger(IdentityController.class);

    private final IdentityRpcProvider identityRpcProvider;
    private final JsonProvider jsonProvider;

    private final UserRpcProvider userRpcProvider;

    private final RetryProvider retryProvider;

    public WechatController(IdentityRpcProvider identityRpcProvider,
                            JsonProvider jsonProvider,
                            UserRpcProvider userRpcProvider, RetryProvider retryProvider) {
        this.identityRpcProvider = identityRpcProvider;
        this.jsonProvider = jsonProvider;
        this.userRpcProvider = userRpcProvider;
        this.retryProvider = retryProvider;
    }

    @ApiOperation(value = "微信小程序授权并单点登入", response = ToCWechatCgiBinSignInResponse.class)
    @PostMapping("/auto-login")
    public ResponseEntity<ToCWechatCgiBinSignInResponse> autoLogin(@Valid @RequestBody ToCWechatCgiBinSignInRequest request, HttpServletRequest httpServletRequest) {
        request.setIpAddress(IPUtils.getIpAddress(httpServletRequest));
        ToCWechatCgiBinSignInResponse response = this.identityRpcProvider.wechatCgiBinSignIn(request);

        return this.ok(response);
    }

    @ApiOperation(value = "执行身份核验;身份核验完成后;自动刷新新的授权登入的token(以此作为替换auto-login的token)", response = SignInResponse.class)
    @PostMapping("/confirm-check-status")
    public ResponseEntity<TocUserConfirmCheckStatusResponse> doConfirmCheckStatus(
            @RequestBody TocUserConfirmCheckStatusRequest request,
            HttpServletRequest httpServletRequest) {
        WechatCgiBinSubject subject = null;
        String cgiToken = ServletRequestUtil.getBearerToken(httpServletRequest);
        String content = null;
        try {
            content = JwtUtil.getContentFromToken(cgiToken);
            subject = this.jsonProvider.toObject(WechatCgiBinSubject.class, content);
        } catch (Exception ex) {
            throw new UnAuthorizedTenantException(String.format("Token(%s;content=%s)无效-认证失败", cgiToken, content));
        }

        /**
         * 身份核验
         */
        String ipAddress = IPUtils.getIpAddress(httpServletRequest);
        TocWechatCgiConfirmCheckStatusRequest tocCheckStatusRequest = TocWechatCgiConfirmCheckStatusRequest.create(
                subject.getTenantUserId(),
                request.getName(),
                request.getCredentialType(),
                request.getNumber(),
                request.getHeadPhoto(),
                request.getFrontPhoto(),
                request.getReversePhoto(),
                request.getAddress(),
                request.getValidDateFrom(),
                request.getValidDateTo(),
                request.getCertificateImage(),
                request.getPhone(),
                ipAddress
        );

        String selectedTenantUserId = null;
        try {
            TocWechatCgiConfirmCheckStatusResponse checkStatusResponse =
                    this.userRpcProvider.confirmCheckStatusForToc(tocCheckStatusRequest);
            selectedTenantUserId = checkStatusResponse.getTenantUserId();
        } catch (ConfirmCheckStatusException ex) {
            selectedTenantUserId = ex.getTenantUserId();
        }
        catch (Exception ex) {
            //todo:
            throw ex;
        }

        /**
         * 绑定用户信息
         */
        if (SystemConstant.isEmpty(subject.getTenantUserId())) {
            WechatCgiBinSubject finalSubject = subject;
            String finalSelectedTenantUserId = selectedTenantUserId;
            this.retryProvider.execute(() -> {
                this.identityRpcProvider.doAssignTocUser(
                        TocUserAssignIdentityRequest.create(
                                finalSelectedTenantUserId,
                                finalSubject.getOpenId(),
                                finalSubject.getUnionId(),
                                request.getName(),
                                request.getPhone(),
                                request.getNumber()
                        )
                );
            }, 5);
        }

        String jsonValue = jsonProvider.getJson(WechatCgiBinSubject.create(subject.getOpenId(), subject.getUnionId(), selectedTenantUserId));
        String newCgiToken = JwtUtil.getMobileToken(jsonValue);
        String token = JwtUtil.getMobileToken(selectedTenantUserId);

        return this.ok(TocUserConfirmCheckStatusResponse.create(newCgiToken, token));
    }
}
