package com.bcxin.rest.web.apis.controllers;

import com.bcxin.api.interfaces.tenants.ExternalGroupRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.externalGroups.ExternalGroupRequest;
import com.bcxin.api.interfaces.tenants.requests.externalGroups.ExternalGroupSearchRequest;
import com.bcxin.api.interfaces.tenants.responses.ExternalGroupResponse;
import io.swagger.annotations.*;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.util.Collection;

/**
 * 团体的外部用户
 */
@Api("当前用户信息")
@RestController
@RequestMapping("/organizations/{organizationId}/external-groups")
public class ExternalGroupController extends ControllerAbstract{

    private final ExternalGroupRpcProvider externalGroupRpcProvider;

    public ExternalGroupController(ExternalGroupRpcProvider externalGroupRpcProvider) {
        this.externalGroupRpcProvider = externalGroupRpcProvider;
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("创建团体分组信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping
    public ResponseEntity post(
            @PathVariable String organizationId,
            @RequestBody ExternalGroupRequest request,
            HttpServletResponse response) throws IOException {
        this.externalGroupRpcProvider.create(request);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("修改团体分组信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PutMapping("/{id}")
    public ResponseEntity put(
            @PathVariable String organizationId,
            @PathVariable String id,
            @RequestBody ExternalGroupRequest request,
            HttpServletResponse response) throws IOException {
        this.externalGroupRpcProvider.update(id, request);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("删除团体分组信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @DeleteMapping("/{id}")
    public ResponseEntity delete(
            @PathVariable String organizationId,
            @PathVariable String id,
            HttpServletResponse response) throws IOException {
        this.externalGroupRpcProvider.delete(id);

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("查询所有团体分组信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/search")
    public ResponseEntity<Collection<ExternalGroupResponse>> search(
            @PathVariable String organizationId,
            @RequestBody ExternalGroupSearchRequest request,
            HttpServletResponse response) throws IOException {
        Collection<ExternalGroupResponse> data = this.externalGroupRpcProvider.search(request);

        return this.ok(data);
    }
}
