package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.api.interfaces.tenants.DepartAdminRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.ContractCriteria;
import com.bcxin.api.interfaces.tenants.criterias.DepartAdminCriteria;
import com.bcxin.api.interfaces.tenants.requests.BatchCreateDepartAdminRequest;
import com.bcxin.api.interfaces.tenants.requests.BatchDeleteDepartAdminRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.UpdateBasicTenantUserRequest;
import com.bcxin.api.interfaces.tenants.requests.uploads.UpdateDepartAdminRequest;
import com.bcxin.api.interfaces.tenants.responses.DepartAdminGetResponse;
import com.bcxin.api.interfaces.tenants.responses.DepartResponse;
import com.bcxin.rest.web.apis.requests.SearchDepartAdminRequest;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/depart-admins")
public class DepartAdminController extends ControllerAbstract {
    private final DepartAdminRpcProvider departAdminRpcProvider;
    private final CacheV2Provider cacheV2Provider;
    private final RetryProvider retryProvider;

    public DepartAdminController(DepartAdminRpcProvider departAdminRpcProvider,CacheV2Provider cacheV2Provider,RetryProvider retryProvider) {
        this.departAdminRpcProvider = departAdminRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
        this.retryProvider = retryProvider;
    }

    @ApiOperation(value = "批量添加部门管理员", response = Void.class)
    @PostMapping("/batch-create")
    public ResponseEntity post(@PathVariable String organizationId,
                               @RequestBody BatchCreateDepartAdminRequest request) {
        this.departAdminRpcProvider.batchCreate(organizationId, request);
        CacheUtil.clearDepartments(cacheV2Provider, organizationId,request.getEmployeeIds());
        CacheUtil.clearCurrentOrganizationsCache(cacheV2Provider);

        return this.ok();
    }
    @ApiOperation(value = "修改部门管理员", response = Void.class)
    @PostMapping("/update")
    public ResponseEntity post(@PathVariable String organizationId,
                               @RequestBody UpdateDepartAdminRequest request) {
        this.departAdminRpcProvider.update(organizationId, request);
        CacheUtil.clearDepartments(cacheV2Provider, organizationId,request.getEmployeeId());
        CacheUtil.clearCurrentOrganizationsCache(cacheV2Provider);

        return this.ok();
    }

    @ApiOperation(value = "批量删除部门管理员", response = Void.class)
    @PostMapping("/batch-delete")
    public ResponseEntity delete(@PathVariable String organizationId,
                                 @RequestBody BatchDeleteDepartAdminRequest request) {
        this.departAdminRpcProvider.batchDelete(organizationId, request);
        CacheUtil.clearDepartments(cacheV2Provider, organizationId,request.getEmployeeIds());
        CacheUtil.clearCurrentOrganizationsCache(cacheV2Provider);

        return this.ok();
    }

    @ApiOperation(value = "搜索部门管理员", response = DepartAdminGetResponse.class, responseContainer = "list")
    @PostMapping("/search")
    public ResponseEntity search(@PathVariable String organizationId,
                                                         @RequestBody SearchDepartAdminRequest request) {
        DepartAdminCriteria criteria = DepartAdminCriteria.create(
                request.getPageIndex(),
                request.getPageSize(),
                organizationId, this.getSelectedEmployeeId(),request.getDepartIds(),request.getAdminDepartIds(),request.getKeyword());

        criteria.changePage(request.getPageIndex(), request.getPageSize());

        Pageable<DepartAdminGetResponse> responsePageable =
                this.departAdminRpcProvider.search(criteria);

        return this.ok(responsePageable);
    }


    @ApiOperation(value = "根据员工id查询管理部门信息", response = DepartResponse.class)
    @GetMapping("/get-by-employee-id/{employeeId}")
    public ResponseEntity getByEmployeeId(@PathVariable String employeeId) {
        List<DepartResponse> departs = this.departAdminRpcProvider.getByEmployeeId(employeeId);
        return this.ok(departs);
    }


    @ApiOperation(value = "导出部门管理员列表")
    @RequestMapping("/exportExcel")
    public void exportExcel(@PathVariable String organizationId,@RequestBody DepartAdminCriteria criteria, HttpServletResponse response) throws Exception {
        String filePath = this.departAdminRpcProvider.exportExcel(organizationId,criteria);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "部门管理员.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }
}
