package com.bcxin.tenant.open.infrastructures.enums;

import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import io.swagger.v3.oas.annotations.media.Schema;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 资源类型: 保安资源/人口资源
 */
@Schema(name = "ResourceType",title = "人员/驻勤点的资源类型")
public enum ResourceType {
    @Schema(name = "Security", title = "保安员/驻勤点")
    Security {
        @Override
        public String getTypeName() {
            return "保安/驻勤点";
        }

        @Override
        public int GetTypeValue() {
            return 1 << 0; //1
        }
    },
    @Schema(name = "Population", title = "社会力量/社区警务")
    Population {
        @Override
        public String getTypeName() {
            return "社会力量/社区警务";
        }

        @Override
        public int GetTypeValue() {
            return 1 << 1;//2
        }
    },

    /**
     * 内保单位的数据: 比如: 甲方管理员
     */
    @Schema(name = "Proprietor", title = "内保单位-甲方管理员")
    Proprietor {
        @Override
        public String getTypeName() {
            return "甲方管理员";
        }

        @Override
        public int GetTypeValue() {
            return 1 << 2;//4
        }
    },
    /**
     * 内保单位管理的数据: 比如: 所关联的保安员和驻勤点
     */
    @Schema(name = "ProprietorManaged", title = "内保单位管理的保安员/驻勤点")
    ProprietorManaged {
        @Override
        public String getTypeName() {
            return "内保单位相关的人员和驻勤点";
        }

        @Override
        public int GetTypeValue() {
            return 1 << 3;//8
        }
    },
    @Schema(name = "TemporarySecurity", title = "临保业务")
    TemporarySecurity {
        @Override
        public String getTypeName() {
            return "临保业务";
        }

        @Override
        public int GetTypeValue() {
            return 1 << 4;//16
        }
    };

    public abstract String getTypeName();

    public abstract int GetTypeValue();

    public static String create(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return ResourceType.Security.name();
        }

        if ("0601".equalsIgnoreCase(institutional)) {
            return ResourceType.Population.name();
        }

        /**
         * 甲方管理员/内保单位
         */
        if (institutional.startsWith("04")) {
            return ResourceType.Proprietor.name();
        }

        return ResourceType.Security.name();
    }

    /**
     * 获取调度台对应的资源信息
     *
     * @param
     * @return
     */
    public static Set<String> getResources(Collection<DeskType> deskTypes) {
        if (deskTypes == null || deskTypes.isEmpty()) {
            return Collections.EMPTY_SET;
        }

        Set<String> resources =
                deskTypes.stream().flatMap(deskType -> {
                    /**
                     * 内保调度台信息
                     */
                    if (deskType == DeskType.Proprietor) {
                        return Set.of(ResourceType.ProprietorManaged.name(), ResourceType.Proprietor.name()).stream();
                    }
                    if(deskType==DeskType.TemporarySecurity){
                        return Set.of(ResourceType.TemporarySecurity.name()).stream();
                    }

                    return Set.of(ResourceType.Security.name(), ResourceType.Population.name()).stream();
                }).collect(Collectors.toSet());

        return resources;
    }

    public static Set<ResourceType> toResourceTypes(Set<String> resourceTypes) {
        if (resourceTypes == null || resourceTypes.isEmpty()) {
            return Collections.EMPTY_SET;
        }

        return resourceTypes.stream().filter(ii -> !StringUtil.isEmpty(ii)).map(ii -> {
            switch (ii) {
                case "Security":
                    return ResourceType.Security;
                case "Population":
                    return ResourceType.Population;
                case "Proprietor":
                    return ResourceType.Proprietor;
                case "ProprietorManaged":
                    return ResourceType.ProprietorManaged;
                case "TemporarySecurity":
                    return ResourceType.TemporarySecurity;
            }
            return null;
        }).filter(ii -> ii != null).collect(Collectors.toSet());
    }

    public static boolean hasResourceType(Set<String> resourceTypes,ResourceType resourceType) {
        if (resourceType == null) {
            return false;
        }

        Set<ResourceType> rts = toResourceTypes(resourceTypes);

        return rts.contains(resourceType);
    }

    public static int getCalculatedResourceValue(Set<ResourceType> resourceTypes) {
        int calculatedValue = 0;
        if (resourceTypes == null) {
            return calculatedValue;
        }

        for (var rs : resourceTypes) {
            calculatedValue = calculatedValue ^ rs.GetTypeValue();
        }

        return calculatedValue;
    }

    public static int getCalculatedResourceValueByDeskTypes(Collection<DeskType> deskTypes) {
        int calculatedValue = 0;
        if (deskTypes == null) {
            return calculatedValue;
        }

        for (var dk : deskTypes.stream().collect(Collectors.toSet())) {
            switch (dk){
                case Normal -> {
                    calculatedValue=calculatedValue^ ResourceType.Security.GetTypeValue();
                }
                case Proprietor -> {
                    calculatedValue=calculatedValue^ ResourceType.Proprietor.GetTypeValue();
                }
                case TemporarySecurity -> {
                    calculatedValue=calculatedValue^ ResourceType.TemporarySecurity.GetTypeValue();
                }
            }
        }

        return calculatedValue;
    }
}
