package com.bcxin.backend.tasks;

import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import com.bcxin.backend.domain.configs.FileModeConfig;
import com.bcxin.backend.core.AppConfigProperty;
import com.bcxin.backend.core.utils.ExceptionUtils;
import com.bcxin.backend.core.utils.FtpUtils;
import com.bcxin.backend.domain.syncs.repositories.DataSyncQueueLogRepository;
import com.bcxin.backend.domain.syncs.repositories.FileSyncQueueRepository;
import com.bcxin.runtime.domain.syncs.entities.FileSyncQueueEntity;
import com.bcxin.saas.core.components.JsonProvider;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collection;

@Component
public class DownloadSyncFileJobTask extends TaskAbstract {
    private final FileSyncQueueRepository fileSyncQueueRepository;
    private final AppConfigProperty appConfigProperty;
    private final JsonProvider jsonProvider;

    public DownloadSyncFileJobTask(DataSyncQueueLogRepository dataSyncLogRepository,
                                   FileSyncQueueRepository fileSyncQueueRepository,
                                   AppConfigProperty appConfigProperty,
                                   JsonProvider jsonProvider) {
        this.fileSyncQueueRepository = fileSyncQueueRepository;
        this.appConfigProperty = appConfigProperty;
        this.jsonProvider = jsonProvider;
    }

    @Override
    protected void runCore() {
        Collection<FileSyncQueueEntity> fileSyncQueueEntities = null;
        Page<FileSyncQueueEntity> pagedFileSyncQueueEntities =
                this.fileSyncQueueRepository.getPendings(Pageable.ofSize(100));
        fileSyncQueueEntities = pagedFileSyncQueueEntities.getContent();

        log(String.format("共有%s个data sync log待处理;", fileSyncQueueEntities.size()));

        if (fileSyncQueueEntities.size() > 0) {
            for (FileSyncQueueEntity fileSyncQueueEntity : fileSyncQueueEntities) {
                try {
                    this.downloadFile(fileSyncQueueEntity);
                } catch (Exception ex) {
                    logger.error(String.format("id=%s; msg=%s", fileSyncQueueEntity.getId(), ex.toString()), ex);
                }
            }
            this.fileSyncQueueRepository.saveAll(fileSyncQueueEntities);
        }
    }

    @Override
    public String getTaskName() {
        return "sync.download.file";
    }

    private void downloadFile(FileSyncQueueEntity fileSyncQueueEntity) {
        StringBuilder sb = new StringBuilder("DownloadSyncFileJobTask.downloadFile;");
        boolean isSuccess = true;
        try {
            sb.append(String.format("准备下载文件资源: %s;", fileSyncQueueEntity.getDownloadUrl()));
            InputStream inputStream = new URL(fileSyncQueueEntity.getDownloadUrl()).openStream();
            boolean flag = checkUrlFileExists(inputStream);
            if (flag && FileModeConfig.getUseFtp()
//                    && fileSyncQueueEntity.getPath().indexOf("/uploads/item/") <0
                    && fileSyncQueueEntity.getPath().indexOf("/uploads/lib/") < 0) {
                sb.append(String.format("即将上传到FTP:%s", fileSyncQueueEntity.getPath()));

                isSuccess = FtpUtils.bcxUploadFile(FileModeConfig.getHost(),
                        FileModeConfig.getUserName(),
                        FileModeConfig.getPassword(),
                        FileModeConfig.getPort(),
                        (StrUtil.isEmpty(FileModeConfig.getFilePath()) ? "" : FileModeConfig.getFilePath()) + fileSyncQueueEntity.getPath(),
                        new URL(fileSyncQueueEntity.getDownloadUrl()).openStream());
            } else if (flag) {
                String destinationPath = appConfigProperty.getRootPath().concat(fileSyncQueueEntity.getPath());
                HttpUtil.downloadFile(fileSyncQueueEntity.getDownloadUrl(), destinationPath);
                sb.append(String.format("即将下载到:%s", destinationPath));
                sb.append("完成文件下载");
            } else {
                isSuccess = false;
                sb.append("文件下载失败：图片不存在或者0Kb");
            }
            log(sb.toString());
        } catch (Exception ex) {
            isSuccess = false;
            sb.insert(0, String.format("文件下载异常: %s", ExceptionUtils.getStackMessage(ex)));
            throw new RuntimeException(sb.toString(), ex);
        } finally {
            logger.error(sb.toString());
            fileSyncQueueEntity.makeStatus(isSuccess, sb.toString());
        }
    }

    public boolean checkUrlFileExists(InputStream inputStream) {
        boolean flag = false;
        try {
            byte[] bytes = new byte[1024];
            if (inputStream.read(bytes) != -1) {
                flag = true;
            }
            return flag;
        } catch (Exception e) {
            return false;
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    logger.error("inputStream关流异常，{}", e);
                }
            }
        }
    }
}
